#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#

from abc import abstractmethod
from xld.kubernetes import get_client_for_api_version
from kubernetes import client
from kubernetes.client import V1DeleteOptions
from xld.kubernetes.core_api_client import KubernetesCoreClient


class Resource(object):
    def __init__(self, container, api_version='v1'):
        self.k8s_client = KubernetesCoreClient(container.container)
        self.api_version = api_version

    def get_api_client(self):
        print ("[Using API version: {}]".format(self.api_version))
        return get_client_for_api_version(self.k8s_client.internal_api_client, self.api_version)

    @abstractmethod
    def create(self, namespace, resource_definition):
        pass

    @abstractmethod
    def modify(self, namespace, resource_definition):
        pass

    @abstractmethod
    def delete(self, namespace, resource_definition):
        pass

    @abstractmethod
    def filter_resources_by_definition(self, namespace, resource_definition):
        pass


class PodResourceProvider(Resource):
    def create(self, namespace, resource_definition):
        response = self.get_api_client().create_namespaced_pod(
            body=resource_definition,
            namespace=namespace
        )
        return response

    def modify(self, namespace, resource_definition):
        response = self.get_api_client().patch_namespaced_pod(
            body=resource_definition,
            name=resource_definition["metadata"]["name"],
            namespace=namespace
        )
        return response

    def delete(self, namespace, resource_definition):
        body = V1DeleteOptions()
        response = self.get_api_client().delete_namespaced_pod(
            namespace=namespace,
            name=resource_definition["metadata"]["name"],
            body=body
        )
        return response

    def filter_resources_by_definition(self, namespace, resource_definition):
        field_selector = "metadata.name={}".format(resource_definition["metadata"]["name"])
        response = self.get_api_client().list_namespaced_pod(
            namespace=namespace,
            field_selector=field_selector
        )
        return response


class ServiceResourceProvider(Resource):
    def create(self, namespace, resource_definition):
        response = self.get_api_client().create_namespaced_service(
            body=resource_definition,
            namespace=namespace
        )
        return response

    def modify(self, namespace, resource_definition):
        response = self.get_api_client().patch_namespaced_service(
            body=resource_definition,
            name=resource_definition["metadata"]["name"],
            namespace=namespace
        )
        return response

    def delete(self, namespace, resource_definition):
        response = self.get_api_client().delete_namespaced_service(
            namespace=namespace,
            name=resource_definition["metadata"]["name"]
        )
        return response

    def filter_resources_by_definition(self, namespace, resource_definition):
        field_selector = "metadata.name={}".format(resource_definition["metadata"]["name"])
        response = self.get_api_client().list_namespaced_service(
            namespace=namespace,
            field_selector=field_selector
        )
        return response


class DeploymentResourceProvider(Resource):
    def create(self, namespace, resource_definition):
        response = self.get_api_client().create_namespaced_deployment(
            body=resource_definition,
            namespace=namespace
        )
        return response

    def modify(self, namespace, resource_definition):
        response = self.get_api_client().patch_namespaced_deployment(
            body=resource_definition,
            name=resource_definition["metadata"]["name"],
            namespace=namespace
        )
        return response

    def delete(self, namespace, resource_definition):
        body = client.V1DeleteOptions(orphan_dependents=False, grace_period_seconds=0)
        response = self.get_api_client().delete_namespaced_deployment(
            namespace=namespace,
            name=resource_definition["metadata"]["name"],
            body=body
        )
        return response

    def filter_resources_by_definition(self, namespace, resource_definition):
        field_selector = "metadata.name={}".format(resource_definition["metadata"]["name"])
        response = self.get_api_client().list_namespaced_deployment(
            namespace=namespace,
            field_selector=field_selector
        )
        return response


class StatefulSetResourceProvider(Resource):
    def create(self, namespace, resource_definition):
        response = self.get_api_client().create_namespaced_stateful_set(
            body=resource_definition,
            namespace=namespace
        )
        return response

    def modify(self, namespace, resource_definition):
        response = self.get_api_client().patch_namespaced_stateful_set(
            body=resource_definition,
            name=resource_definition["metadata"]["name"],
            namespace=namespace
        )
        return response

    def delete(self, namespace, resource_definition):
        body = client.V1DeleteOptions(orphan_dependents=False, grace_period_seconds=0)
        response = self.get_api_client().delete_namespaced_stateful_set(
            namespace=namespace,
            name=resource_definition["metadata"]["name"],
            body=body
        )
        return response

    def filter_resources_by_definition(self, namespace, resource_definition):
        field_selector = "metadata.name={}".format(resource_definition["metadata"]["name"])
        response = self.get_api_client().list_namespaced_stateful_set(
            namespace=namespace,
            field_selector=field_selector
        )
        return response


class SecretResourceProvider(Resource):
    def create(self, namespace, resource_definition):
        response = self.get_api_client().create_namespaced_secret(
            namespace=namespace,
            body=resource_definition)
        return response

    def modify(self, namespace, resource_definition):
        response = self.get_api_client().replace_namespaced_secret(
            name=resource_definition["metadata"]["name"],
            namespace=namespace,
            body=resource_definition)
        return response

    def delete(self, namespace, resource_definition):
        body = client.V1DeleteOptions(orphan_dependents=False, grace_period_seconds=0)
        response = self.get_api_client().delete_namespaced_secret(name=resource_definition["metadata"]["name"],
                                                                    namespace=namespace,
                                                                    body=body)
        return response

    def filter_resources_by_definition(self, namespace, resource_definition):
        field_selector = "metadata.name={}".format(resource_definition["metadata"]["name"])
        response = self.get_api_client().list_namespaced_secret(namespace=namespace,
                                                                  field_selector=field_selector)

        return response


class ConfigMapResourceProvider(Resource):
    def create(self, namespace, resource_definition):
        response = self.get_api_client().create_namespaced_config_map(
            namespace=namespace,
            body=resource_definition)
        return response

    def modify(self, namespace, resource_definition):
        response = self.get_api_client().replace_namespaced_config_map(name=resource_definition["metadata"]["name"],
                                                                         namespace=namespace,
                                                                         body=resource_definition)
        return response

    def delete(self, namespace, resource_definition):
        body = client.V1DeleteOptions(orphan_dependents=False, grace_period_seconds=0)
        response = self.get_api_client().delete_namespaced_config_map(name=resource_definition["metadata"]["name"],
                                                                        namespace=namespace,
                                                                        body=body)
        return response

    def filter_resources_by_definition(self, namespace, resource_definition):
        field_selector = "metadata.name={}".format(resource_definition["metadata"]["name"])
        response = self.get_api_client().list_namespaced_config_map(namespace=namespace,
                                                                      field_selector=field_selector)

        return response


class PersistentVolumeResourceProvider(Resource):
    def create(self, namespace, resource_definition):
        response = self.get_api_client().create_persistent_volume(
            body=resource_definition
        )
        return response

    def modify(self, namespace, resource_definition):
        response = self.get_api_client().patch_persistent_volume(
            body=resource_definition,
            name=resource_definition["metadata"]["name"]
        )
        return response

    def delete(self, namespace, resource_definition):
        body = client.V1DeleteOptions()
        response = self.get_api_client().delete_persistent_volume(
            name=resource_definition["metadata"]["name"],
            body=body
        )
        return response

    def filter_resources_by_definition(self, namespace, resource_definition):
        field_selector = "metadata.name={}".format(resource_definition["metadata"]["name"])
        response = self.get_api_client().list_persistent_volume(
            field_selector=field_selector
        )
        return response


class PersistentVolumeClaimResourceProvider(Resource):
    def create(self, resource_definition, namespace= None):
        response = self.get_api_client().create_namespaced_persistent_volume_claim(
            body=resource_definition,
            namespace=namespace
        )
        return response

    def modify(self, namespace, resource_definition):
        response = self.get_api_client().patch_namespaced_persistent_volume_claim(
            body=resource_definition,
            name=resource_definition["metadata"]["name"],
            namespace=namespace
        )
        return response

    def delete(self, namespace, resource_definition):
        body = client.V1DeleteOptions()
        response = self.get_api_client().delete_namespaced_persistent_volume_claim(
            namespace=namespace,
            name=resource_definition["metadata"]["name"],
            body=body
        )
        return response

    def filter_resources_by_definition(self, namespace, resource_definition):
        field_selector = "metadata.name={}".format(resource_definition["metadata"]["name"])
        response = self.get_api_client().list_namespaced_service(
            namespace=namespace,
            field_selector=field_selector
        )
        return response


class IngressResourceProvider(Resource):
    def create(self, resource_definition, namespace= None):
        response = self.get_api_client().create_namespaced_ingress(
            body=resource_definition,
            namespace=namespace
        )
        return response

    def modify(self, namespace, resource_definition):
        response = self.get_api_client().patch_namespaced_ingress(
            body=resource_definition,
            name=resource_definition["metadata"]["name"],
            namespace=namespace
        )
        return response

    def delete(self, namespace, resource_definition):
        body = client.V1DeleteOptions()
        response = self.get_api_client().delete_namespaced_ingress(
            namespace=namespace,
            name=resource_definition["metadata"]["name"],
            body=body
        )
        return response

    def filter_resources_by_definition(self, namespace, resource_definition):
        field_selector = "metadata.name={}".format(resource_definition["metadata"]["name"])
        response = self.get_api_client().list_namespaced_ingress(
            namespace=namespace,
            field_selector=field_selector
        )
        return response


class IngressResourceProvider(Resource):
    def create(self, resource_definition, namespace= None):
        response = self.get_api_client().create_namespaced_ingress(
            body=resource_definition,
            namespace=namespace
        )
        return response

    def modify(self, namespace, resource_definition):
        response = self.get_api_client().patch_namespaced_ingress(
            body=resource_definition,
            name=resource_definition["metadata"]["name"],
            namespace=namespace
        )
        return response

    def delete(self, namespace, resource_definition):
        body = client.V1DeleteOptions()
        response = self.get_api_client().delete_namespaced_ingress(
            namespace=namespace,
            name=resource_definition["metadata"]["name"],
            body=body
        )
        return response

    def filter_resources_by_definition(self, namespace, resource_definition):
        field_selector = "metadata.name={}".format(resource_definition["metadata"]["name"])
        response = self.get_api_client().list_namespaced_ingress(
            namespace=namespace,
            field_selector=field_selector
        )
        return response


class StorageClassResourceProvider(Resource):
    def create(self, namespace, resource_definition):
        response = self.get_api_client().create_storage_class(
            body=resource_definition
        )
        return response

    def modify(self, namespace, resource_definition):
        response = self.get_api_client().patch_storage_class(
            body=resource_definition,
            name=resource_definition["metadata"]["name"]
        )
        return response

    def delete(self, namespace, resource_definition):
        body = client.V1DeleteOptions()
        response = self.get_api_client().delete_storage_class(
            name=resource_definition["metadata"]["name"],
            body=body
        )
        return response

    def filter_resources_by_definition(self, namespace, resource_definition):
        response = self.get_api_client().list_storage_class()
        items = []
        for storage_class in response.items:
            if resource_definition["metadata"]["name"] == storage_class.metadata.name:
                items.append(storage_class)
        response.items = items
        return response


class CronJobResourceProvider(Resource):
    def create(self, resource_definition, namespace= None):
        response = self.get_api_client().create_namespaced_cron_job(
            body=resource_definition,
            namespace=namespace
        )
        return response

    def modify(self, namespace, resource_definition):
        response = self.get_api_client().patch_namespaced_cron_job(
            body=resource_definition,
            name=resource_definition["metadata"]["name"],
            namespace=namespace
        )
        return response

    def delete(self, namespace, resource_definition):
        body = client.V1DeleteOptions()
        response = self.get_api_client().delete_namespaced_cron_job(
            namespace=namespace,
            name=resource_definition["metadata"]["name"],
            body=body
        )
        return response

    def filter_resources_by_definition(self, namespace, resource_definition):
        field_selector = "metadata.name={}".format(resource_definition["metadata"]["name"])
        response = self.get_api_client().list_namespaced_cron_job(
            namespace=namespace,
            field_selector=field_selector
        )
        return response
