#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#

import json
from org.apache.commons.io import IOUtils

import yaml
from xld.kubernetes.factories.handler_factory import ContainerHelperFactory
from xld.kubernetes.factories.handler_factory import ResourceFactoryResolver


class ResourceHelper(object):
    def __init__(self, deployed):
        self.__deployed = deployed
        self.__file_data = deployed.file.getInputStream()
        self.__container_helper = ContainerHelperFactory(deployed.container).create()

    def validate_resources(self):
        valid, unsupported_resources = ResourceFactoryResolver(self.__deployed) \
            .get_factory() \
            .validate_resource_types(ResourceHelper(self.__deployed).parse())

        if not valid:
            raise RuntimeError(self.__get_error_msg(unsupported_resources))

    def parse(self):
        data = self.__get_data()
        try:
            json_data = json.loads(data)
            return json_data['items'] if json_data['kind'] == 'List' else [json_data]
        except:
            try:
                return list(yaml.safe_load_all(data))
            except:
                raise

    def __get_data(self):
        try:
            data = IOUtils.toString(self.__file_data, "UTF-8")
            return data
        finally:
            IOUtils.closeQuietly(self.__file_data)

    def __get_error_msg(self, unsupported_resources):
        if bool(unsupported_resources):
            msg = "Unsupported resource {0} for {1} {2}".format(unsupported_resources[0],
                                                                self.__container_helper.get_container_label(),
                                                                self.__container_helper.get_container_name(
                                                                    self.__deployed.container))
        else:
            msg = "Unsupported resource for {0} {1}".format(self.__container_helper.get_container_label(),
                                                            self.__container_helper.get_container_name(
                                                                self.__deployed.container))
        return msg
