from com.xebialabs.deployit.plugin.api.deployment.planning import Checkpoint
from com.xebialabs.deployit.plugin.api.deployment.specification import Operation
from xld.kubernetes.factories.handler_factory import ResourceFactoryResolver
from xld.kubernetes.resource.helper import ResourceHelper
import itertools


class StepsGenerator(object):
    def __init__(self, context, steps, delta):
        self.__context = context
        self.__steps = steps
        self.__delta = delta
        self.__last_step = None

    def generate(self, action, order, data, wait_details, checkpoint_name, operation):
        checkpoint = Checkpoint(self.__delta._delegate, checkpoint_name, operation)

        self.__last_step = self.__steps.jython(
            description="{0} {1} {2}".format(action, data['kind'], data['metadata']['name']),
            script="xld/kubernetes/resource/steps/{0}.py".format(action.lower()),
            jython_context={"data": data},
            order=order
        )
        self.__context.addStepWithCheckpoint(self.__last_step, checkpoint)

        self.__context.addStep(self.__steps.jython(
            description="Wait for {0} {1} to be {2}".format(data['kind'], data['metadata']['name'],
                                                            wait_details['action']),
            script="xld/kubernetes/resource/steps/{0}.py".format(wait_details['script']),
            jython_context={"data": data},
            order=order + 1
        ))

    def should_execute_step(self, checkpoint_name):
        checkpoints = self.__delta.intermediateCheckpoints
        print "Intermediate checkpoints {0}".format(checkpoints)
        return not checkpoints or checkpoint_name in checkpoints

    def done(self):
        if self.__last_step:
            self.__context.addCheckpoint(self.__last_step, self.__delta._delegate)


class PlanGenerator(object):
    def __init__(self, deployed, steps_generator):
        self._resource_factory = ResourceFactoryResolver(deployed).get_factory()
        self.__resource_helper = ResourceHelper(deployed)
        self.__steps_generator = steps_generator

    def generate(self):
        pass

    def _generate(self, action):
        step_order_dict = self._resource_factory.get_resource_order()
        wait_details = self._resource_factory.get_resource_wait_details()
        data = self.__resource_helper.parse()
        items = data['items'] if data['kind'] == 'List' else [data]
        self._generate_steps(action, wait_details, step_order_dict, items, None)
        self.__steps_generator.done()

    def _generate_steps(self, action, wait_details, step_order_dict, items, operation):
        step_index = 0
        for item in items:
            step_index += 1
            checkpoint_name = self.__checkpoint_name(item, step_index)
            if self.__steps_generator.should_execute_step(checkpoint_name):
                order = step_order_dict[item['kind']][action]
                self.__steps_generator.generate(action, order, item, self.__get_wait_detail(wait_details[action], item['kind']), checkpoint_name, operation)

    @staticmethod
    def __checkpoint_name(item, step_index):
        return "{0}_{1}".format(PlanGenerator._resource_fq_name(item), step_index)

    @staticmethod
    def _resource_fq_name(item):
        return "{0}_{1}".format(item['kind'], item['metadata']['name'])

    @staticmethod
    def __get_wait_detail(wait_details, kind):
        return wait_details[kind] if kind in wait_details else wait_details['Default']


class CreatePlanGenerator(PlanGenerator):
    def __init__(self, deployed, steps_generator):
        super(CreatePlanGenerator, self).__init__(deployed, steps_generator)

    def generate(self):
        self._generate("Create")


class DestroyPlanGenerator(PlanGenerator):
    def __init__(self, deployed, steps_generator):
        super(DestroyPlanGenerator, self).__init__(deployed, steps_generator)

    def generate(self):
        self._generate("Destroy")


class ModifyPlanGenerator(PlanGenerator):
    def __init__(self, previous_deployed, deployed, steps_generator):
        super(ModifyPlanGenerator, self).__init__(deployed, steps_generator)
        self.__previous_resource_helper = ResourceHelper(previous_deployed)

    def generate(self):
        self._generate("Modify")

    def __to_map_of_list(self, items):
        groupby = itertools.groupby(items, lambda item: item['kind'])
        list = [(kind, map(lambda item: self._resource_fq_name(item), item_list)) for kind, item_list in groupby]
        resources_by_kind = dict((key, value) for key, value in list)
        return resources_by_kind

    def _generate_steps(self, action, wait_details, step_order_dict, items, operation):
        previous_data = self.__previous_resource_helper.parse()
        previous_items = previous_data['items'] if previous_data['kind'] == 'List' else [previous_data]
        current_resource_map = self.__to_map_of_list(items)
        previous_resource_map = self.__to_map_of_list(previous_items)

        current_kinds = set(current_resource_map.keys())
        previous_kinds = set(previous_resource_map.keys())

        kinds = current_kinds | previous_kinds
        for kind in kinds:
            current_resource_names = set(current_resource_map[kind]) if kind in current_kinds else set()
            previous_resource_names = set(previous_resource_map[kind]) if kind in previous_kinds else set()

            delete_resource_names = previous_resource_names - current_resource_names
            new_resource_names = current_resource_names - previous_resource_names
            modify_resource_names = current_resource_names & previous_resource_names

            if kind in self._resource_factory.get_recreate_resources():
                delete_resource_names |= modify_resource_names
                new_resource_names |= modify_resource_names
                modify_resource_names = set()

            self.__build_steps(delete_resource_names, new_resource_names, modify_resource_names, previous_items, items,
                               operation, action)

    def __build_steps(self, delete_resource_names, new_resource_names, modify_resource_names, previous_items, items,
                      operation, action):
        delete_items = filter(lambda item: self._resource_fq_name(item) in delete_resource_names, previous_items)
        new_items = filter(lambda item: self._resource_fq_name(item) in new_resource_names, items)
        modify_items = filter(lambda item: self._resource_fq_name(item) in modify_resource_names, items)

        step_order_dict = self._resource_factory.get_resource_order()
        wait_details = self._resource_factory.get_resource_wait_details()

        super(ModifyPlanGenerator, self)._generate_steps("Destroy", wait_details, step_order_dict, delete_items, Operation.DESTROY)
        super(ModifyPlanGenerator, self)._generate_steps(action, wait_details, step_order_dict, modify_items, operation)
        super(ModifyPlanGenerator, self)._generate_steps("Create", wait_details, step_order_dict, new_items, Operation.CREATE)
