from kubernetes import client

from xld.kubernetes.commons.common_utils import CommonUtils
from xld.kubernetes.persistent_volume.service import PersistentVolumeService


class PVCHelper(object):
    def __init__(self, deployed):
        self.__deployed = deployed

    def read_pvc(self):
        pvc = client.V1PersistentVolumeClaim()
        pvc.metadata = self.__read_metadata()

        spec = client.V1PersistentVolumeClaimSpec()
        spec.volume_name = self.__deployed.volumeName
        spec.access_modes = self.__deployed.accessModes
        if bool(self.__deployed.matchExpressions or self.__deployed.matchLabels):
            spec.selector = self.__read_selector()

        if bool(self.__deployed.limits or self.__deployed.requests):
            spec.resources = self.__read_resources()
        pvc.spec = spec
        return pvc

    def __read_metadata(self):
        metadata = client.V1ObjectMeta(name=self.get_name(self.__deployed))
        if bool(self.__deployed.storageClassName):
            annotations = {}
            annotations['volume.beta.kubernetes.io/storage-class'] = self.__deployed.storageClassName
            metadata.annotations = annotations
        return metadata

    def __read_resources(self):
        resources = client.V1ResourceRequirements()
        resources.limits = self.__get_capacity_dict(self.__deployed.limits)
        resources.requests = self.__get_capacity_dict(self.__deployed.requests)
        return resources

    @staticmethod
    def get_name(deployed):
        return deployed.claimName if deployed.claimName else deployed.name

    @staticmethod
    def __read_match_expression(deployed_match_expression):
        match_expression = client.UnversionedLabelSelectorRequirement()
        match_expression.key = deployed_match_expression.key
        match_expression.operator = deployed_match_expression.operator
        if deployed_match_expression.matchValues:
            match_expression.values = deployed_match_expression.matchValues
        return match_expression

    def __read_selector(self):
        selector = client.UnversionedLabelSelector()
        if self.__deployed.matchExpressions:
            selector.match_expressions = []
            for deployed_match_expression in self.__deployed.matchExpressions:
                selector.match_expressions.append(PVCHelper.
                                                  __read_match_expression(deployed_match_expression))
        if self.__deployed.matchLabels:
            selector.match_labels = self.__deployed.matchLabels

        return selector

    def should_not_modify_name(self, previous_deployed):
        if PVCHelper.get_name(previous_deployed) != PVCHelper.get_name(self.__deployed):
            raise RuntimeError('PersistentVolumeClaim name should not be modified.')

    def should_not_modify_volume(self, previous_deployed):
        if previous_deployed.volumeName != self.__deployed.volumeName:
            raise RuntimeError('Reference to the binding PersistentVolume should not be modified.')

    def should_check_retain_policy(self):
        persistent_volume_service = PersistentVolumeService(self.__deployed)
        if self.__deployed.volumeName and persistent_volume_service.is_existing(self.__deployed.volumeName):
            volume = persistent_volume_service.fetch(self.__deployed.volumeName)
            if volume.spec.persistent_volume_reclaim_policy == 'Retain':
                raise RuntimeError('PersistentVolumeClaim name should not be modified as the binding PersistentVolume '
                                   'has Retain as reclaim policy.')

    @staticmethod
    def __get_capacity_dict(storage_value):
        capacity_prop = {}
        capacity_prop['storage'] = storage_value
        return capacity_prop
