import importlib


class K8SResourceFactory(object):
    def __init__(self, deployed):
        self.__deployed = deployed

    def is_supported(self, data):
        unsupported_resources = filter(lambda item: item not in self._get_supported_resources(),
                                       map(lambda item: item['kind'], data['items'])) if \
            data['kind'] == 'List' else (
            [data['kind']] if data['kind'] not in self._get_supported_resources() else None)
        return self._is_valid_resource_type(data), unsupported_resources

    @staticmethod
    def get_resource_order():
        return {
            'Pod': {'Create': 60,'Modify': 50,'Destroy': 40},
            'Service': {'Create': 67,'Modify': 55,'Destroy': 45},
            'Secret': {'Create': 67,'Modify': 55,'Destroy': 45},
            'ConfigMap': {'Create': 67,'Modify': 55,'Destroy': 45},
            'Deployment': {'Create': 62,'Modify': 53,'Destroy': 42},
            'PersistentVolume': {'Create': 56,'Modify': 46,'Destroy': 45},
            'PersistentVolumeClaim': {'Create': 58,'Modify': 48,'Destroy': 43}
        }

    @staticmethod
    def get_recreate_resources():
        return ['PersistentVolumeClaim']

    @staticmethod
    def get_resource_wait_details():
        return {
            "Create": {
                "Default": {'script': 'create_update_wait', 'action': "created"},
                "Pod": {'script': 'create_update_wait', 'action': "in running state"},
                "Deployment": {'script': 'deployment/create_update_wait', 'action': "in running state"},
                "PersistentVolumeClaim": {'script': 'create_update_wait', 'action': "in Bound phase"}
            },
            "Destroy": {
                "Default": {'script': 'delete_wait', 'action': "destroyed completely"}
            },
            "Modify": {
                "Default": {'script': 'create_update_wait', 'action': "modified"},
                "Pod": {'script': 'create_update_wait', 'action': "in running state"},
                "Deployment": {'script': 'deployment/create_update_wait', 'action': "in running state"}
            }
        }

    def get(self, data):
        return self._resolve(data)

    def _resolve(self, data):
        clazz = "{0}ResourceProvider".format(data["kind"])
        factory_module = importlib.import_module("xld.kubernetes.resource.provider")
        provider_clazz = getattr(factory_module, clazz)
        instance = provider_clazz(self.__deployed.container)
        return instance

    def _is_valid_resource_type(self, data):
        return reduce(lambda acc, cur: bool(acc and (cur in self._get_supported_resources())),
                      map(lambda item: item['kind'], data['items'])) \
            if data['kind'] == 'List' else  data['kind'] in self._get_supported_resources()

    @staticmethod
    def _get_supported_resources():
        return K8SResourceFactory.get_resource_order().keys()
