from kubernetes import client
from kubernetes.client.api_client import ApiClient

from commons.common_utils import CommonUtils


class ServiceHelper(object):
    def read_service(self, deployed_service):
        service = client.V1Service()
        service.metadata = client.V1ObjectMeta(name=self.get_service_name(deployed_service))
        service.metadata.resource_version = deployed_service.resourceVersion

        spec = client.V1ServiceSpec()
        spec.type = deployed_service.serviceType
        spec.cluster_ip = deployed_service.clusterIP
        spec.external_i_ps = deployed_service.externalIPs
        spec.external_name = deployed_service.externalName
        spec.load_balancer_ip = deployed_service.loadBalancerIP
        spec.session_affinity = deployed_service.sessionAffinity

        if deployed_service.selectors:
            spec.selector = deployed_service.selectors

        if deployed_service.loadBalancerSourceRanges:
            spec.load_balancer_source_ranges = deployed_service.loadBalancerSourceRanges

        if deployed_service.ports:
            spec.ports = self.__read_ports(deployed_service)

        spec = ApiClient().sanitize_for_serialization(spec)
        service.spec = spec
        service = ApiClient().sanitize_for_serialization(service)
        return service

    @staticmethod
    def get_service_name(deployed):
        return deployed.serviceName if deployed.serviceName else deployed.name

    @staticmethod
    def __read_ports(deployed_service):
        ports = []
        for deployed_port in deployed_service.ports:
            ports.append(ServiceHelper.__read_port(deployed_port))
        return ports

    @staticmethod
    def __read_port(deployed_port):
        port = client.V1ServicePort()
        port.name = deployed_port.portName if deployed_port.portName else deployed_port.name
        port.port = deployed_port.port
        port.node_port = deployed_port.nodePort
        port.protocol = deployed_port.protocol

        if deployed_port.targetPort:
            port.target_port = CommonUtils.convert_to_int_if_possible(deployed_port.targetPort)
        return port

    @staticmethod
    def read_output_properties(service_request, previous_deployed):
        metadata = service_request['metadata']
        metadata['resourceVersion'] = previous_deployed.resourceVersion
        service_request['metadata'] = metadata

        spec = service_request['spec']
        if ('clusterIP' not in spec) and previous_deployed.outputClusterIP and spec['type'] != 'ExternalName':
            spec['clusterIP'] = previous_deployed.outputClusterIP
        if ('externalIPs' not in spec) and previous_deployed.outputExternalIPs:
            spec['externalIPs'] = previous_deployed.outputExternalIPs
        if ('loadBalancerIP' not in spec) and previous_deployed.outputLoadBalancerIP:
            spec['loadBalancerIP'] = previous_deployed.outputLoadBalancerIP
        service_request['spec'] = spec

    @staticmethod
    def update_output_properties(deployed, service_response):
        deployed.serviceName = service_response.metadata.name
        deployed.resourceVersion = service_response.metadata.resource_version
        deployed.outputClusterIP = service_response.spec.cluster_ip
        deployed.outputExternalIPs = service_response.spec.external_i_ps
        deployed.outputLoadBalancerIP = service_response.spec.load_balancer_ip
        return deployed
