from abc import abstractmethod

from kubernetes import client
from kubernetes.client import V1DeleteOptions
from xld.kubernetes.core_api_client import KubernetesCoreClient


class Resource(object):
    def __init__(self, container):
        self.k8s_client = KubernetesCoreClient(container.container)

    def get_k8s_core_api(self):
        return client.CoreV1Api(self.k8s_client.internal_api_client)

    def get_k8s_apps_api(self):
        return client.ExtensionsV1beta1Api(self.k8s_client.internal_api_client)

    @abstractmethod
    def create(self, namespace, resource_definition):
        pass

    @abstractmethod
    def modify(self, namespace, resource_definition):
        pass

    @abstractmethod
    def delete(self, namespace, resource_definition):
        pass

    @abstractmethod
    def filter_resources_by_definition(self, namespace, resource_definition):
        pass


class PodResourceProvider(Resource):
    def create(self, namespace, resource_definition):
        response = self.get_k8s_core_api().create_namespaced_pod(
            body=resource_definition,
            namespace=namespace
        )
        return response

    def modify(self, namespace, resource_definition):
        response = self.get_k8s_core_api().patch_namespaced_pod(
            body=resource_definition,
            name=resource_definition["metadata"]["name"],
            namespace=namespace
        )
        return response

    def delete(self, namespace, resource_definition):
        body = V1DeleteOptions()
        response = self.get_k8s_core_api().delete_namespaced_pod(
            namespace=namespace,
            name=resource_definition["metadata"]["name"],
            body=body
        )
        return response

    def filter_resources_by_definition(self, namespace, resource_definition):
        field_selector = "metadata.name={}".format(resource_definition["metadata"]["name"])
        response = self.get_k8s_core_api().list_namespaced_pod(
            namespace=namespace,
            field_selector=field_selector
        )
        return response


class ServiceResourceProvider(Resource):
    def create(self, namespace, resource_definition):
        response = self.get_k8s_core_api().create_namespaced_service(
            body=resource_definition,
            namespace=namespace
        )
        return response

    def modify(self, namespace, resource_definition):
        response = self.get_k8s_core_api().patch_namespaced_service(
            body=resource_definition,
            name=resource_definition["metadata"]["name"],
            namespace=namespace
        )
        return response

    def delete(self, namespace, resource_definition):
        response = self.get_k8s_core_api().delete_namespaced_service(
            namespace=namespace,
            name=resource_definition["metadata"]["name"]
        )
        return response

    def filter_resources_by_definition(self, namespace, resource_definition):
        field_selector = "metadata.name={}".format(resource_definition["metadata"]["name"])
        response = self.get_k8s_core_api().list_namespaced_service(
            namespace=namespace,
            field_selector=field_selector
        )
        return response


class DeploymentResourceProvider(Resource):
    def create(self, namespace, resource_definition):
        response = self.get_k8s_apps_api().create_namespaced_deployment(
            body=resource_definition,
            namespace=namespace
        )
        return response

    def modify(self, namespace, resource_definition):
        response = self.get_k8s_apps_api().patch_namespaced_deployment(
            body=resource_definition,
            name=resource_definition["metadata"]["name"],
            namespace=namespace
        )
        return response

    def delete(self, namespace, resource_definition):
        body = client.V1DeleteOptions(orphan_dependents=False, grace_period_seconds=0)
        response = self.get_k8s_apps_api().delete_namespaced_deployment(
            namespace=namespace,
            name=resource_definition["metadata"]["name"],
            body=body
        )
        return response

    def filter_resources_by_definition(self, namespace, resource_definition):
        field_selector = "metadata.name={}".format(resource_definition["metadata"]["name"])
        response = self.get_k8s_apps_api().list_namespaced_deployment(
            namespace=namespace,
            field_selector=field_selector
        )
        return response
