from kubernetes import client
from kubernetes.client.api_client import ApiClient
import cPickle as pk


class PodHelper(object):
    def read_pod(self, deployed_pod, pod=client.V1Pod()):
        self.__apply_properties(pod, deployed_pod)
        pod_dict = ApiClient().sanitize_for_serialization(pod)
        volumes = PodHelper.__get_volume_dict(deployed_pod)
        pod_dict['spec']['volumes'] = volumes
        return pod_dict

    @staticmethod
    def get_pod_name(deployed):
        return deployed.podName if deployed.podName else deployed.name

    @staticmethod
    def __read_container(deployed_container):
        container = client.V1Container()
        container.image = deployed_container.image
        container.command = deployed_container.command
        container.args = deployed_container.args
        container.name = deployed_container.containerName
        return container

    @staticmethod
    def __read_container_port(deployed_port):
        port = client.V1ContainerPort()
        port.host_port = deployed_port.hostPort
        port.container_port = deployed_port.containerPort
        port.protocol = deployed_port.protocol
        port.host_ip = deployed_port.hostIP
        return port

    @staticmethod
    def __read_volume_mount(deployed_volume_mount):
        volume_mount = client.V1VolumeMount()
        volume_mount.mount_path = deployed_volume_mount.mountPath
        volume_mount.name = deployed_volume_mount.volumeName if bool(
            deployed_volume_mount.volumeName) else deployed_volume_mount.name
        volume_mount.read_only = deployed_volume_mount.readOnly
        volume_mount.sub_path = deployed_volume_mount.subPath
        return volume_mount

    def __apply_properties(self, pod, deployed_pod):
        pod.metadata = client.V1ObjectMeta(name=self.get_pod_name(deployed_pod))
        if deployed_pod.labels:
            pod.metadata.labels = deployed_pod.labels

        spec = client.V1PodSpec()
        spec.host_network = deployed_pod.hostNetwork
        spec.containers = []
        for deployed_container in deployed_pod.containers:
            container = self.__read_container(deployed_container)
            if deployed_container.ports:
                container.ports = []
                for deployed_port in deployed_container.ports:
                    port = self.__read_container_port(deployed_port)
                    container.ports.append(port)

            if deployed_container.volumeBindings:
                container.volume_mounts = []
                for volume_binding in deployed_container.volumeBindings:
                    volume_mount = self.__read_volume_mount(volume_binding)
                    container.volume_mounts.append(volume_mount)
            spec.containers.append(container)
        pod.spec = spec

    @staticmethod
    def __get_volume_dict(deployed_pod):
        volumes = []
        for deployed_volume in deployed_pod.volumes:
            volume = {'name': deployed_volume.volumeName if bool(deployed_volume.volumeName) else deployed_volume.name}
            vol_prop = {}
            for key, val in deployed_volume.properties.items():
                vol_prop[key] = True if val.lower() == 'true' else pk.loads(pk.dumps(val))

            volume[deployed_volume.volumeType] = vol_prop
            volumes.append(volume)
        return volumes
