import socket

setattr(socket, 'SOL_TCP', 6)

from kubernetes.config.kube_config import KubeConfigLoader
from kubernetes.client import ApiClient
from commons.common_utils import CommonUtils
from com.xebialabs.deployit.plugin.kubernetes import LoaderUtil
from java.nio.file import Files, Paths, StandardCopyOption
import tempfile
import base64
import os
from kubernetes.client.configuration import ConfigurationObject


class KubernetesClientFactory:
    def __init__(self, container):
        self.__container = container
        self.__context = "kb_context"
        self.__cluster = "kb_cluster"
        self.__user = "kb_user"

    def create(self):
        cluster = self.__create_cluster()
        user = self.__create_user()
        context = {"cluster": self.__cluster, "user": self.__user}
        config_params = self.__get_config_params(cluster=cluster, context=context, user=user)
        return ApiClient(config=self.__create_config(config_params))

    def __create_cluster(self):
        cluster = {"server": self.__container.apiServerURL,
                   'insecure-skip-tls-verify': "true" if self.__container.skipTLS else None,
                   "certificate-authority-data": base64.b64encode(
                       self.__container.caCert) if self.__container.caCert else None}
        return CommonUtils.remove_none_keys(cluster)

    def __create_user(self):
        user = {
            "client-certificate-data": base64.b64encode(self.__container.tlsCert) if self.__container.tlsCert else None,
            "client-key-data": base64.b64encode(
                self.__container.tlsPrivateKey) if self.__container.tlsPrivateKey else None,
            "token": self.__container.token, "username": self.__container.username,
            "password": self.__container.password}
        return CommonUtils.remove_none_keys(user)

    def __get_config_params(self, context, cluster, user):
        return {"current-context": self.__context,
                "contexts": [{"name": self.__context, "context": context}],
                "clusters": [{"name": self.__cluster, "cluster": cluster}],
                "users": [{"name": self.__user, "user": user}]}

    def __create_config(self, config_params):
        client_config = ConfigurationObject()
        config_loader = KubeConfigLoader(config_dict=config_params, active_context=self.__context,
                                         client_configuration=client_config)
        config_loader.load_and_set()
        if self.__container.skipTLS:
            self.__disable_https_warnings()
            client_config.ssl_ca_cert = self.__set_ca_bundle_path(path="certifi/cacert.pem")
        return client_config

    @staticmethod
    def __disable_https_warnings():
        import urllib3
        urllib3.disable_warnings()

    def __set_ca_bundle_path(self, path):
        ca_bundle_path = self.__extract_file_from_jar(path)
        os.environ['REQUESTS_CA_BUNDLE'] = ca_bundle_path
        return ca_bundle_path

    def __extract_file_from_jar(self, config_file):
        file_url = LoaderUtil.getResourceBySelfClassLoader(config_file)
        if file_url:
            tmp_file = None
            try:
                tmp_file, tmp_abs_path = tempfile.mkstemp()
                Files.copy(file_url.openStream(), Paths.get(tmp_abs_path), StandardCopyOption.REPLACE_EXISTING)
                return tmp_abs_path
            finally:
                if tmp_file:
                    tmp_file.close()
        else:
            return None
