#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#

import json

from com.xebialabs.deployit.plugin.api.reflect import Type

from java.io import BufferedReader
from java.io import InputStreamReader
from java.io import ByteArrayInputStream
from java.lang import StringBuilder
from java.lang import RuntimeException
from java.lang import String
from org.apache.commons.io import IOUtils

import yaml
from xld.kubernetes.factories.handler_factory import ContainerHelperFactory
from xld.kubernetes.yaml.dumper import yaml_safe_dump

class ResourceHelper(object):
    def __init__(self, deployed, context, stitch = None):
        self.__context = context
        self.__deployed = deployed
        self.__file_data = deployed.json if hasattr(deployed, 'json') and deployed.json else self.__get_data(deployed.file.getInputStream())
        self.__container_helper = ContainerHelperFactory(deployed.container).create()
        self.__stitch = stitch

    def parse(self):
        items, json_data = self.get_items()
        if self.__stitch is not None and self.__deployed.type.instanceOf(Type.valueOf("k8s.AbstractResources")):
            try:
                transformed_items = []
                items =  self.additional_stitch(items, "preProcessingTransformation", "Pre-processing transformation") + items
                for item in items:
                    item_name = "{0}_{1}".format(item['kind'], item['metadata']['name'])
                    try:
                        invocation_name = "Transform {0} {1}".format(self.__deployed.name, item['kind'])
                        transformation_context = {"format": "yaml", "invocationName": invocation_name}
                        transformed_content = self.__stitch(ByteArrayInputStream(String(yaml_safe_dump(item)).getBytes()), transformation_context)
                        stringified = self.stringify(transformed_content)
                        transformed_items.append(yaml.safe_load(stringified))
                        print('Template transformation applied for {}'.format(item_name))
                    except RuntimeException as e:
                        print("Planning script could not apply template transformation for {}. Reason: {}".format(item_name, e.getMessage()))
                        transformed_items.append(item)
                transformed_items.extend(self.additional_stitch(items, "postProcessingTransformation", "Post-processing transformation"))
                return transformed_items, json_data
            except RuntimeException as e:
                print("Planning script could not apply template transformation. Reason: {}".format(e.getMessage()))
                return items, json_data
        else:
            return items, json_data

    def get_items(self):
        data = self.__file_data
        try:
            # applying patches
            modified_str = self.__context.patch(data, self.__deployed.deployable)
            patched_json = json.loads(modified_str)
            return patched_json['items'] if patched_json['kind'] == 'List' else [patched_json], True
        except:
            try:
                yaml_list = list(yaml.safe_load_all(data))
                items = []
                for item in yaml_list:

                    # applying patches
                    item_stream = yaml.safe_dump(item)
                    modified_stream = self.__context.patch(item_stream, self.__deployed.deployable)
                    patched_item = yaml.safe_load(modified_stream)
                    if patched_item:
                        if patched_item['kind'] == 'List':
                            for sub_item in patched_item['items']:
                                items.append(sub_item)
                        else:
                            items.append(patched_item)
                return items, False
            except:
                raise

    @staticmethod
    def __get_data(filestream):
        try:
            data = IOUtils.toString(filestream, "UTF-8")
            return data
        finally:
            IOUtils.closeQuietly(filestream)

    def __get_error_msg(self, unsupported_resources):
        if bool(unsupported_resources):
            msg = "Unsupported resource {0} for {1} {2}".format(unsupported_resources[0],
                                                                self.__container_helper.get_container_label(),
                                                                self.__container_helper.get_container_name(
                                                                    self.__deployed.container))
        else:
            msg = "Unsupported resource for {0} {1}".format(self.__container_helper.get_container_label(),
                                                            self.__container_helper.get_container_name(
                                                                self.__deployed.container))
        return msg

    def additional_stitch(self, items, key, name):
        stitch_processing_input = yaml.safe_dump(items)
        try:
            stitch_processing_output = self.stringify(self.__stitch(ByteArrayInputStream(stitch_processing_input), {"format": "yaml", key: "true", "invocationName": name}))
            return list(yaml.safe_load(stitch_processing_output))
        except RuntimeException as e:
            print("Planning script could not apply {} for {}. Reason: {}".format(key, stitch_processing_input, e.getMessage()))
            return []

    def stringify(self, stream):
        reader = BufferedReader(InputStreamReader(stream))
        out = StringBuilder()

        while True:
            line = reader.readLine()
            if line is None:
                break
            out.append(line)
            out.append("\n")

        reader.close()
        return out.toString()
