#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#

from xld.kubernetes.commons.common_utils import CommonUtils
from xld.kubernetes.core_api_client import KubernetesCoreClient
from xld.kubernetes.persistent_volume.helper import PersistentVolumeHelper
from kubernetes import client as kubernetes_client
from kubernetes.client.rest import ApiException


class PersistentVolumeService(object):

    def __init__(self, deployed):
        self.__persistent_volume_helper = PersistentVolumeHelper()
        self.__deployed = deployed
        self.__container = deployed.container.container if hasattr(deployed.container, 'container') else deployed.container
        self.__client = KubernetesCoreClient(self.__container)
        self.__kind = CommonUtils.get_kind(self)

    def create(self):
        with self.__client as client:
            persistent_volume_request = self.__persistent_volume_helper.read_persistent_volume(self.__deployed)
            print("Creating PersistentVolume '{0}' ".format(PersistentVolumeHelper.get_name(self.__deployed)))
            CommonUtils.print_request_dump(self.__container, persistent_volume_request, self.__kind,
                                           client.get_api_version())
            persistent_volume_response = client.core_api.create_persistent_volume(body=persistent_volume_request)
        CommonUtils.print_response_dump(self.__container, persistent_volume_response)
        return persistent_volume_response

    def modify(self):
        with self.__client as client:
            persistent_volume_request = self.__persistent_volume_helper.read_persistent_volume(self.__deployed)
            print("Updating PersistentVolume '{0}' ".format(PersistentVolumeHelper.get_name(self.__deployed)))
            CommonUtils.print_request_dump(self.__container, persistent_volume_request, self.__kind,
                                           client.get_api_version())
            persistent_volume_response = client.core_api.replace_persistent_volume(
                name=PersistentVolumeHelper.get_name(self.__deployed), body=persistent_volume_request)
        CommonUtils.print_response_dump(self.__container, persistent_volume_response)
        return persistent_volume_response

    def destroy(self):
        with self.__client as client:
            print("Destroying PersistentVolume '{0}' ".format(PersistentVolumeHelper.get_name(self.__deployed)))
            client.core_api.delete_persistent_volume(name=PersistentVolumeHelper.get_name(self.__deployed),
                                                                  body=kubernetes_client.V1DeleteOptions())

    def should_wait_for_creation(self):
        if not self.is_existing(PersistentVolumeHelper.get_name(self.__deployed)):
            print("Waiting for PersistentVolume {0} to be created. ".format(PersistentVolumeHelper.get_name(
                self.__deployed)))
            return True
        return False

    def should_wait_for_destroy(self):
        if self.is_existing(PersistentVolumeHelper.get_name(self.__deployed)):
            print("Waiting for PersistentVolume {0} to be destroyed. ".format(PersistentVolumeHelper.get_name(
                self.__deployed)))
            return True
        return False

    def is_existing(self, name):
        try:
            persistent_volume = self.fetch(name)
            if persistent_volume:
                return True
        except ApiException as e:
            if e.status == 404:
                return False

    def fetch(self, name):
        with self.__client as client:
            persistent_volume = client.core_api.read_persistent_volume(name=name)
        return persistent_volume
