#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#

from kubernetes import client
from kubernetes.client.api_client import ApiClient


class StorageClassHelper(object):

    def __init__(self, deployed):
        self.__deployed = deployed

    def read_storage_class(self):
        storage_class = client.V1StorageClass(provisioner=self.__deployed.provisioner)
        storage_class.metadata = self.__read_metadata()
        if self.__deployed.parameters:
            storage_class.parameters = self.__deployed.parameters
        storage_class.reclaim_policy = self.__deployed.reclaimPolicy
        storage_class.mount_options = self.__deployed.mountOptions
        storage_class.allow_volume_expansion = self.__deployed.allowVolumeExpansion
        storage_class = ApiClient().sanitize_for_serialization(storage_class)
        return storage_class

    def __read_metadata(self):
        metadata = client.V1ObjectMeta(name=self.get_name(self.__deployed))
        if self.__deployed.labels:
            metadata.labels = self.__deployed.labels
        if self.__deployed.annotations:
            metadata.annotations = self.__deployed.annotations
        return metadata

    @staticmethod
    def get_name(deployed):
        return deployed.storageClassName if deployed.storageClassName else deployed.name

    def should_not_modify_name(self, previous_deployed):
        if self.get_name(previous_deployed) != self.get_name(self.__deployed):
            raise RuntimeError('StorageClass name should not be modified.')

    def should_not_modify_provisioner(self, previous_deployed):
        if previous_deployed.provisioner != self.__deployed.provisioner:
            raise RuntimeError('StorageClass provisioner should not be modified.')

    def should_not_modify_parameters(self, previous_deployed):
        if previous_deployed.parameters != self.__deployed.parameters:
            raise RuntimeError('StorageClass parameters should not be modified.')

    def should_not_modify_reclaim_policy(self, previous_deployed):
        if previous_deployed.reclaimPolicy != self.__deployed.reclaimPolicy:
            raise RuntimeError('StorageClass reclaim policy should not be modified.')
