#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#
import textwrap
from pprint import pprint

from xld.kubernetes.apps_api_client import KubernetesAppsClient
from xld.kubernetes.core_api_client import KubernetesCoreClient
from xld.kubernetes.factories.handler_factory import ContainerHelperFactory


class DescribeHelper(object):
    def __init__(self, deployed):
        self.__deployed = deployed
        self.__container_helper = ContainerHelperFactory(deployed.container).create()

    def describe_pod(self, name, kind):
        client = KubernetesCoreClient(self.__deployed.container.container)
        namespace = self.__container_helper.get_container_name(self.__deployed.container)
        pod_response = client.describe_pod(name=name, namespace=namespace,
                                           show_container_logs=self.__deployed.showContainerLogs,
                                           bytes_container_logs=self.__deployed.bytesToReadFromContainerLogs)
        if not pod_response:
            print("Pods not created, describing {0} '{1}':".format(kind, name))
            apps_client = KubernetesAppsClient(self.__deployed.container.container)
            if kind == "StatefulSet":
                statefulset = apps_client.read_statefulset(name, namespace)
                self.describe_controller(statefulset, kind, client.list_events(statefulset.metadata.uid, namespace))
            elif kind == "Deployment":
                deployment = apps_client.read_deployment(name, namespace)
                self.describe_controller(deployment, kind, client.list_events(deployment.metadata.uid, namespace))
            else:
                print("Describing {0} is not supported.".format(kind))
        else:
            for res in pod_response:
                data = res['data']
                if data:
                    metadata = data.metadata
                    spec = data.spec
                    status = data.status
                    print(textwrap.dedent("""
                        ---------------------------------------
                        Describing Pod {0}
                        ---------------------------------------
                        """.format(res['name'])))
                    # Print pod data
                    cmd = textwrap.dedent("""
                            Name:           {0}
                            Namespace:      {1}
                            Priority:       {2}
                            Node:           {3}/{4}
                            Start Time:     {5}
                            Labels:         {6}
                            Status:         {7}
                            IP:             {8}
                            Containers:
                            """.format(
                        metadata.name, metadata.namespace, spec.priority, spec.node_name, status.host_ip,
                        status.start_time,
                        self.get_labels(metadata.labels), status.phase, status.pod_ip
                    ))

                    for container in spec.containers:
                        container_state = self.get_container_state(container, status)
                        cmd += textwrap.dedent("""
                              - {0}:
                                  Image:          {1}
                                  Ports:          {2}
                                  State:          {3}
                                  Ready:          {4}
                                  Restart Count:  {5}
                                  Limits:         {6}
                                  Requests:       {7}
                                  Environment:    {8}
                                  Mounts:
                                    {9}
                            """.format(
                            container.name, container.image, self.get_ports(container), self.get_state(container_state),
                            self.get_ready_state(container_state), self.get_restart_count_state(container_state),
                            self.get_limits(container.resources.limits), self.get_limits(container.resources.requests),
                            container.env, self.get_volume_mounts(container.volume_mounts)
                        ))

                    cmd += textwrap.dedent("""
                            Conditions:
                        """)

                    for container in status.conditions:
                        cmd += textwrap.dedent("""
                                  -  {0}: {1}
                            """.format(container.type, container.status))

                    cmd += textwrap.dedent("""
                            Volumes:
                        """)

                    for v in spec.volumes:
                        cmd += textwrap.dedent("""
                              - {0}
                                """.format(v.name))

                    cmd += textwrap.dedent("""

                            Events:
                                [Type - Reason](From) : Message
                                -------------------------------
                        """)

                    # Print events
                    events = res['events']
                    for event in events.items:
                        cmd += textwrap.dedent("""
                            -   [{0} - {1}]({2}): {3} 
                            """.format(event.type, event.reason, event.source.component, event.message))
                    print(cmd)
                    print("\n")
                    # Print container logs
                    if res['logs']:
                        print(textwrap.dedent("""
                                ---------------------------------------
                                Container logs for pod {0}
                                ---------------------------------------
                                """.format(res['name'])))
                        for log in res['logs']:
                            if log:
                                pprint(log)

    def describe_controller(self, data, kind, events):
        metadata = data.metadata
        spec = data.spec
        status = data.status

        print(textwrap.dedent("""
                        ---------------------------------------
                        Describing {0} {1}
                        ---------------------------------------
                        """.format(kind, metadata.name)))
        print(textwrap.dedent("""
                            Name:               {0}
                            Namespace:          {1}
                            CreationTimestamp:  {2}
                            Selector:           {3}
                            Labels:             {4}
                            """.format(metadata.name, metadata.namespace,
                                       metadata.creation_timestamp.strftime('%a, %d %b %Y %H:%M:%S'),
                                       self.get_labels(spec.selector.match_labels),
                                       self.get_labels(metadata.labels))))

        self.print_annotations(metadata.annotations)
        update_strategy_type = ""
        if "update_strategy" in spec:
            update_strategy_type = spec.update_strategy.type
        print(textwrap.dedent("""
                            Replicas:           {0} desired / {1} total
                            Update Strategy:    {2}
                              Partition:        {3}
                            """.format(self.get_number(spec.replicas),
                                       self.get_number(status.ready_replicas), update_strategy_type,
                                       self.nested_get(spec, ['update_strategy', 'rolling_update', 'partition']))))

        template = spec.template

        print(textwrap.dedent("""
                                Pod Template:
                                  Labels:          {0}
                                """.format(self.get_labels(template.metadata.labels))))
        print("  Init Containers:\n")
        for init_container in template.spec.init_containers:
            self.print_container(init_container)

        print("  Containers:\n")
        for container in template.spec.containers:
            self.print_container(container)

        print("  Volumes:\n")
        for volume in template.spec.volumes:
            print("   {}".format(volume.name))

        print("Volume Claims:\n")
        for volume_claim in spec.volume_claim_templates:
            print("- Name:          {}".format(volume_claim.metadata.name))
            print("  StorageClass:  {}".format(volume_claim.spec.storage_class_name))
            print("  Labels:        {}".format(self.get_labels(volume_claim.metadata.labels)))
            self.print_annotations(volume_claim.metadata.annotations, "  Annotations:   ")
            print("  Capacity:      {}".format(
                self.nested_get(volume_claim, ['spec', 'resources', 'requests', 'storage'])))
            print("  AccessModes:   [{}]".format(' '.join(volume_claim.spec.access_modes)))

        print("Events:")
        if len(events.items) > 0:
            from_column_len = len(events.items[0].source.component) + 3
            aligned_line = "  {0:<15}{1:<14}{2:<" + str(from_column_len) + "}{3:<30}\n"
            events_txt = aligned_line.format("Type", "Reason", "From", "Message") + \
                         aligned_line.format("____", "______", "____", "_______")
            for event in events.items:
                events_txt += aligned_line.format(event.type, event.reason, event.source.component,
                                                  event.message)
            print(events_txt)
        else:
            print("  No events recorded.")

        print("---------------------------------------")


    def print_annotations(self, annotations, init_text="Annotations:        "):
        if annotations:
            text = init_text
            indentation = len(text)
            total = len(annotations)
            index = 1
            for k, v in annotations.items():
                text += "{0}: {1}\n".format(k, v)
                if index < total:
                    text += " " * indentation
                index += 1
            print(text)
        else:
            print(init_text)

    def print_container(self, container):
        print('\n'.join(("   {0}:".format(container.name),
                         "    Image:       {0}".format(container.image),
                         "    Ports:       {0}".format(self.get_ports(container)),
                         "    Host Ports:  {0}".format(self.get_ports(container, 'host_port')))))
        print("    Command:\n")
        for command in container.command:
            print("      {0}\n".format(command))
        if container.args:
            print("    Args:\n")
            for arg in container.args:
                print("      {0}\n".format(arg))

        if container.resources.limits:
            print("    Limits:\n")
            for k, v in container.resources.limits.items():
                if v:
                    print("      {0}: {1}\n".format(k, v))

        if container.resources.requests:
            print("    Requests:\n")
            for k, v in container.resources.requests.items():
                if v:
                    print("      {0}: {1}\n".format(k, v))

        print("    Environment:  {0}\n".format(container.env))

        print("    Mounts:\n")
        for volume_mount in container.volume_mounts:
            print("      {0} from {1}\n".format(volume_mount.mount_path, volume_mount.name))

    @staticmethod
    def get_number(number):
        if isinstance(number, int):
            return number
        else:
            return 0

    @staticmethod
    def nested_get(input_dict, nested_key):
        internal_dict_value = input_dict
        if not isinstance(input_dict, dict):
            internal_dict_value = input_dict.to_dict()
        for k in nested_key:
            internal_dict_value = internal_dict_value.get(k, None)
            if internal_dict_value is None:
                return None
        return internal_dict_value

    @staticmethod
    def get_container_state(container, status):
        if status:
            if status.container_statuses:
                for cstat in status.container_statuses:
                    if cstat.name == container.name and container.image in cstat.image:
                        return cstat
        return ""

    @staticmethod
    def get_ready_state(container_state):
        out = ""
        if hasattr(container_state, 'ready'):
            if container_state.ready:
                out = "True"
            else:
                out = "False"
        return out

    @staticmethod
    def get_restart_count_state(container_state):
        out = ""
        if hasattr(container_state, 'restart_count'):
            out = container_state.restart_count
        return out

    @staticmethod
    def get_labels(labels):
        out = ""
        if labels:
            for k, v in labels.items():
                if v:
                    out += "{0}={1}, ".format(k, v)
        return out

    @staticmethod
    def get_ports(container, port_property="container_port"):
        out = ""
        if hasattr(container, 'ports') and container.ports:
            for port in container.ports:
                if not isinstance(port, dict):
                    port = port.to_dict()
                out += "{0}/{1}, ".format(port[port_property], port['protocol'])
        return out

    @staticmethod
    def get_state(container_state):
        out = ""
        if hasattr(container_state, 'state'):
            state = container_state.state
            if state.running and hasattr(state.running, 'reason'):
                out += "running: {0} ({1})\n".format(state.running.reason, state.running.message)
            if state.terminated and hasattr(state.terminated, 'reason'):
                out += "terminated: {0} ({1})\n".format(state.terminated.reason, state.terminated.message)
            if state.waiting and hasattr(state.waiting, 'reason'):
                out += "waiting: {0} ({1})\n".format(state.waiting.reason, state.waiting.message)
        return out

    @staticmethod
    def get_limits(limits):
        out = ""
        if limits:
            for k, v in limits.items():
                if v:
                    out += "{0}: {1}, ".format(k, v)
        return out

    @staticmethod
    def get_volume_mounts(mounts):
        out = ""
        if mounts:
            for port in mounts:
                out += "{0} from {1}\n".format(port.mount_path, port.name)
        return out
