#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#

from com.xebialabs.deployit.plugin.api.deployment.planning import Checkpoint
from com.xebialabs.deployit.plugin.api.deployment.specification import Operation
from xld.kubernetes.factories.handler_factory import ResourceFactoryResolver
from xld.kubernetes.factories.handler_factory import ContainerHelperFactory
from xld.kubernetes.resource.helper import ResourceHelper
import itertools
import json
from deepdiff import DeepDiff


class StepsGenerator(object):
    def __init__(self, context, steps, delta, container):
        self.__context = context
        self.__steps = steps
        self.__delta = delta
        self.__last_step = None
        self.__container = container
        self.__container_helper = ContainerHelperFactory(self.__container).create()

    def generate(self, action, order, data, wait_details, checkpoint_name, operation, json_data=True):
        checkpoint = Checkpoint(self.__delta._delegate, checkpoint_name, operation)

        container_name = self.__container_helper.get_container_name(self.__container)
        self.__last_step = self.__steps.jython(
            description="{0} {1} {2} on {3}".format(action, data['kind'], data['metadata']['name'], container_name),
            script="xld/kubernetes/resource/steps/{0}.py".format(action.lower()),
            jython_context={"data": data, "json_data": json_data},
            order=order,
            preview_script="xld/kubernetes/resource/steps/preview.py".format(wait_details['script'])
        )
        self.__context.addStepWithCheckpoint(self.__last_step, checkpoint)

        self.__context.addStep(self.__steps.jython(
            description="Wait for {0} {1} to be {2} on {3}".format(data['kind'], data['metadata']['name'],
                                                                   wait_details['action'], container_name),
            script="xld/kubernetes/resource/steps/{0}.py".format(wait_details['script']),
            jython_context={"data": data, 'action': action},
            order=order + 1
        ))

    def should_execute_step(self, checkpoint_name):
        checkpoints = self.__delta.intermediateCheckpoints
        print "Intermediate checkpoints {0}".format(checkpoints)
        return not checkpoints or checkpoint_name in checkpoints

    def done(self):
        if self.__last_step:
            self.__context.addCheckpoint(self.__last_step, self.__delta._delegate)


class PlanGenerator(object):
    def __init__(self, deployed, steps_generator, context, stitch = None):
        self.deployed = deployed
        self._resource_factory = ResourceFactoryResolver(deployed).get_factory()
        self.__resource_helper = ResourceHelper(deployed, context, stitch)
        self.__steps_generator = steps_generator

    def generate(self):
        pass

    def update_params(self, template_body):
        new_params = self.deployed.parameters
        for param in template_body.get('parameters', ''):
            if param['name'] in new_params:
                if 'generate' in param:
                    del param['generate']
                param['value'] = new_params[param['name']]

    def _generate(self, action):
        step_order_dict = self._resource_factory.get_resource_order()
        wait_details = self._resource_factory.get_resource_wait_details()
        items, json_data = self.__resource_helper.parse()
        if self.deployed.type == 'openshift.TemplateResources':
            self.update_params(items[0])
        self._generate_steps(action, wait_details, step_order_dict, items, None, json_data)
        self.__steps_generator.done()

    def _generate_steps(self, action, wait_details, step_order_dict, items, operation, json_data):
        step_index = 0
        for item in items:
            step_index += 1
            checkpoint_name = self.__checkpoint_name(item, step_index)
            if self.__steps_generator.should_execute_step(checkpoint_name):
                kind = item['kind']
                self._create_steps(kind, action, step_order_dict, item, wait_details, checkpoint_name, operation, json_data)

            if self.deployed.type == 'openshift.TemplateResources':
                for object in item["objects"]:
                    step_index += 1
                    checkpoint_name = self.__checkpoint_name(object, step_index)
                    if self.__steps_generator.should_execute_step(checkpoint_name):
                        kind = object['kind']
                        self._create_steps(kind, action, step_order_dict, object, wait_details, checkpoint_name, operation, json_data)

    def _create_steps(self, kind, action, step_order_dict, item, wait_details, checkpoint_name, operation, json_data):
        order = step_order_dict[kind][action] if kind in step_order_dict else step_order_dict['Default'][action]
        self.__steps_generator.generate(action, order, item, self.__get_wait_detail(wait_details[action], kind),
                                        checkpoint_name, operation, json_data)

    @staticmethod
    def __checkpoint_name(item, step_index):
        return "{0}_{1}".format(PlanGenerator._resource_fq_name(item), step_index)

    @staticmethod
    def _resource_fq_name(item):
        return "{0}_{1}".format(item['kind'], item['metadata']['name'])

    @staticmethod
    def __get_wait_detail(wait_details, kind):
        return wait_details[kind] if kind in wait_details else wait_details['Default']


class CreatePlanGenerator(PlanGenerator):
    def __init__(self, deployed, steps_generator, context, stitch = None):
        super(CreatePlanGenerator, self).__init__(deployed, steps_generator, context, stitch)

    def generate(self):
        self._generate("Create")


class DestroyPlanGenerator(PlanGenerator):
    def __init__(self, deployed, steps_generator, context, stitch = None):
        super(DestroyPlanGenerator, self).__init__(deployed, steps_generator, context, stitch)

    def generate(self):
        self._generate("Destroy")


class ModifyPlanGenerator(PlanGenerator):
    def __init__(self, previous_deployed, deployed, steps_generator, context, stitch = None):
        super(ModifyPlanGenerator, self).__init__(deployed, steps_generator, context, stitch)
        self.__previous_resource_helper = ResourceHelper(previous_deployed, context, stitch)

    def generate(self):
        self._generate("Modify")

    def __to_map_of_list(self, items):
        groupby = {}
        for item in items:
            if item is not None:
                if item['kind'] in groupby.keys():
                    groupby[item['kind']].append(item)
                else:
                    groupby[item['kind']] = [item]
        list = [(kind, map(lambda item: self._resource_fq_name(item), item_list)) for kind, item_list in
                groupby.items()]
        resources_by_kind = dict((key, value) for key, value in list)
        return resources_by_kind

    def __index_by_item_fq(self, items):
        return {self._resource_fq_name(items[i]): items[i] for i in range(0, len(items))}

    def determine_modified_resources(self, current_resources_index, previous_resources_index,
                                     potential_modify_resource_names):
        modified_resource_names = set()
        for resource_name in potential_modify_resource_names:
            cr = current_resources_index[resource_name]
            pr = previous_resources_index[resource_name]
            ddiff = DeepDiff(cr, pr)
            if len(ddiff.keys()):
                modified_resource_names.add(resource_name)
        return modified_resource_names

    def is_resource_modified(self, previous_resource, current_resource):
        ddiff = DeepDiff(current_resource, previous_resource)
        return len(ddiff.keys())

    def _generate_steps(self, action, wait_details, step_order_dict, items, operation, json_data):
        previous_items, json_data = self.__previous_resource_helper.parse()
        current_resources_index = self.__index_by_item_fq(items)
        current_resources_names = current_resources_index.keys()
        previous_resources_index = self.__index_by_item_fq(previous_items)
        previous_resources_names = previous_resources_index.keys()

        new_resource_names = set()
        modify_resource_names = set()
        delete_resource_names = set()

        for item in items:
            item_name = self._resource_fq_name(item)
            if item_name in previous_resources_names:
                if self.is_resource_modified(previous_resources_index[item_name], item):
                    modify_resource_names.add(item_name)
            else:
                new_resource_names.add(item_name)

        for item in previous_items:
            item_name = self._resource_fq_name(item)
            if item_name not in current_resources_names:
                delete_resource_names.add(item_name)

        self.__build_steps(delete_resource_names, new_resource_names, modify_resource_names, previous_items, items,
                           operation, action, json_data)

    def __build_steps(self, delete_resource_names, new_resource_names, modify_resource_names, previous_items, items,
                      operation, action, json_data):
        delete_items = filter(lambda item: self._resource_fq_name(item) in delete_resource_names, previous_items)
        new_items = filter(lambda item: self._resource_fq_name(item) in new_resource_names, items)
        modify_items = filter(lambda item: self._resource_fq_name(item) in modify_resource_names, items)

        step_order_dict = self._resource_factory.get_resource_order()
        wait_details = self._resource_factory.get_resource_wait_details()

        super(ModifyPlanGenerator, self)._generate_steps("Destroy", wait_details, step_order_dict, delete_items,
                                                         Operation.DESTROY, json_data)
        super(ModifyPlanGenerator, self)._generate_steps(action, wait_details, step_order_dict, modify_items, operation,
                                                         json_data)
        super(ModifyPlanGenerator, self)._generate_steps("Create", wait_details, step_order_dict, new_items,
                                                         Operation.CREATE, json_data)
