#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#
from __future__ import print_function
import time
from xld.kubernetes.factories.handler_factory import ContainerHelperFactory

from xld.kubernetes.factories.handler_factory import ResourceFactoryResolver


class ResourceService:
    def __init__(self, deployed):
        self.__deployed = deployed
        self.__container_helper = ContainerHelperFactory(deployed.container).create()
        self.__resource_factory = ResourceFactoryResolver(deployed).get_factory()

    def create(self, item):
        print("Creating resource {0} on {1} {2}".format(item['kind'], self.__container_helper.get_container_name(
            self.__deployed.container), self.__deployed.container))
        with self.__resource_factory.get(item) as resource:
            resource.create(namespace=self.__container_helper.get_container_name(self.__deployed.container),
                            resource_definition=item)

    def update(self, item):
        print("Updating resource {0} on {1}".format(item['kind'], self.__container_helper.get_container_name(
            self.__deployed.container), self.__deployed.container))
        with self.__resource_factory.get(item) as resource:
            resource.modify(namespace=self.__container_helper.get_container_name(self.__deployed.container),
                            resource_definition=item, patch_type=self.__deployed.mergePatchType,
                            update_method=self.__deployed.updateMethod)

    def delete(self, item):
        print("Deleting resource {0} on {1}".format(item['kind'], self.__container_helper.get_container_name(
            self.__deployed.container), self.__deployed.container))
        with self.__resource_factory.get(item) as resource:
            resource.delete(namespace=self.__container_helper.get_container_name(self.__deployed.container),
                            resource_definition=item, propagation_policy=self.__deployed.propagationPolicy)

    def is_resource_deleted(self, item):
        with self.__resource_factory.get(item) as resource_provider:
            resources = resource_provider.filter_resources_by_definition(
                namespace=self.__container_helper.get_container_name(self.__deployed.container),
                resource_definition=item)
        return 'items' not in dir(resources) or not bool(resources.items)

    def wait_for_deleted(self, item, max_retries):
        resource_provider = self.__resource_factory.get(item)

        retry_countdown = max_retries
        ready = False
        sleep_seconds = 5
        while not ready and retry_countdown > 0:
            resources = resource_provider.filter_resources_by_definition(
                namespace=self.__container_helper.get_container_name(self.__deployed.container),
                resource_definition=item)
            if 'items' not in dir(resources) or not bool(resources.items):
                print("Resource deleted.")
                ready = True
            else:
                print("Waiting for resource to be deleted, retries left: {0} (Approx {1} seconds)".format(
                    retry_countdown, retry_countdown * 5))
                retry_countdown -= 1
                time.sleep(sleep_seconds)

