from xlddocker import connect_to_docker
from docker.errors import ImageNotFound

class CopyImageHelper(object):
    def __init__(self, params, this_ci):
        self.source_params, self.target_params = CopyImageHelper.__build_params(params)
        self.dkr = connect_to_docker(this_ci)

    @staticmethod
    def __build_params(params):
        source_params = dict()
        source_params['repo_url'] = CopyImageHelper.__get_repo_url(params.getProperty("sourceRepoUrl"))
        source_params['secure_registry'] = CopyImageHelper.__is_secure(params.getProperty("sourceRepoUrl"))
        source_params['image_name'], source_params['image_tag'] = CopyImageHelper.__get_image_details(
            params.getProperty("sourceImageNameAndTag"), '', 'latest')
        source_params['user_name'] = params.getProperty("sourceUserName")
        source_params['password'] = params.getProperty("sourcePassword")
        target_params = dict()
        target_params['repo_url'] = CopyImageHelper.__get_repo_url(params.getProperty("targetRepoUrl"))
        target_params['secure_registry'] = CopyImageHelper.__is_secure(params.getProperty("targetRepoUrl"))
        target_params['image_name'], target_params['image_tag'] = CopyImageHelper.__get_image_details(
            params.getProperty("targetImageNameAndTag"), source_params['image_name'], source_params['image_tag'])
        target_params['user_name'] = params.getProperty("targetUserName")
        target_params['password'] = params.getProperty("targetPassword")
        return source_params, target_params

    @staticmethod
    def __is_secure(repo_url):
        return repo_url.startswith("https://")

    @staticmethod
    def __get_repo_url(repo_url):
        if repo_url.startswith("http://"):
            return repo_url[7:]
        elif repo_url.startswith("https://"):
            return repo_url[8:]
        else:
            return repo_url

    @staticmethod
    def __get_image_details(image_name_and_tag, default_image, default_tag):
        splits = image_name_and_tag.split(":") if image_name_and_tag else []
        image_name = splits[0].strip() if bool(len(splits) > 0 and splits[0].strip()) else default_image
        image_tag = splits[1].strip() if bool(len(splits) > 1 and splits[1].strip()) else default_tag
        return image_name, image_tag

    @staticmethod
    def __is_non_empty(value):
        return bool(value and value.strip())

    @staticmethod
    def __build_repo_url(params):
        return '{0}/{1}:{2}'.format(params['repo_url'], params['image_name'], params['image_tag'])

    @staticmethod
    def __get_repo_url_for_tag(params):
        return '{0}/{1}'.format(params['repo_url'], params['image_name'])

    @staticmethod
    def __get_credentials(params):
        return {'username': params['user_name'].strip(), 'password': params['password'].strip()}

    def pull_image(self):
        repo_url = self.__build_repo_url(self.source_params)
        local_image = self.get_image_locally(self.source_params)
        if local_image:
            print 'Found image locally, so will not pull from source repo {0}'.format(repo_url)
            return local_image
        else:
            print 'Image not found locally.'
        print 'Pulling image from repo {0}'.format(repo_url)
        if CopyImageHelper.__is_non_empty(self.source_params['user_name']):
            image = self.dkr.images.pull(repo_url, insecure_registry=not self.source_params['secure_registry'],
                                         auth_config=self.__get_credentials(self.source_params))
        else:
            image = self.dkr.images.pull(repo_url, insecure_registry=not self.source_params['secure_registry'])
        return image

    def add_image_tag(self, image):
        repo_url = self.__get_repo_url_for_tag(self.target_params)
        print 'Adding tag {0} to image {1}.'.format(self.target_params['image_tag'], self.target_params['image_name'])
        image.tag(repo_url, tag=self.target_params['image_tag'])

    def push_image(self):
        repo_url = self.__build_repo_url(self.target_params)
        print 'Pushing image to repo {0}'.format(repo_url)
        if CopyImageHelper.__is_non_empty(self.target_params['user_name']):
            self.dkr.images.push(repo_url, insecure_registry=not self.target_params['secure_registry'],
                                 auth_config=self.__get_credentials(self.target_params))
        else:
            self.dkr.images.push(repo_url, insecure_registry=not self.target_params['secure_registry'])

    def get_image_locally(self, image_params):
        try:
            return self.dkr.images.get(name='{0}/{1}:{2}'.format(image_params['repo_url'], image_params['image_name'], image_params['image_tag']))
        except ImageNotFound:
            return None
