/**
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS
 * FOR A PARTICULAR PURPOSE. THIS CODE AND INFORMATION ARE NOT SUPPORTED BY XEBIALABS.
 */
package com.xebialabs.deployit.plugin.codepipeline.services;

import com.amazonaws.services.codepipeline.AWSCodePipelineClient;
import com.amazonaws.services.codepipeline.model.*;
import com.xebialabs.deployit.plugin.codepipeline.exceptions.CodePipelinePluginExecutionException;
import com.xebialabs.deployit.plugin.codepipeline.helpers.Constants;
import com.xebialabs.deployit.plugin.codepipeline.helpers.S3Helper;
import com.xebialabs.deployit.plugin.codepipeline.helpers.Validator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.nio.file.Path;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class CodePipelineJobService {

    private static final Logger LOGGER = LoggerFactory.getLogger(CodePipelineJobService.class);
    private final AWSCodePipelineClient codePipelineClient;
    private String version;

    public CodePipelineJobService(final AWSCodePipelineClient codePipelineClient) {
        Validator.notNull(codePipelineClient);

        this.codePipelineClient = codePipelineClient;
    }

    public Job getJob(final String xlDeployServerKey) {
        ActionTypeId actionTypeId = this.getActionTypeId();
        LOGGER.info("PollForJobs for action type {}", actionTypeId);
        final PollForJobsRequest pollForJobsRequest = new PollForJobsRequest();
        pollForJobsRequest.setActionTypeId(actionTypeId);
        Map<String, String> queryParam = new HashMap<>();
        queryParam.put("XL Deploy Server Key", xlDeployServerKey);
        pollForJobsRequest.setQueryParam(queryParam);
        pollForJobsRequest.setMaxBatchSize(1);

        final PollForJobsResult pollForJobsResult = codePipelineClient.pollForJobs(pollForJobsRequest);
        LOGGER.info("Found {} job(s)", pollForJobsResult.getJobs().size());
        if (!pollForJobsResult.getJobs().isEmpty())
            return pollForJobsResult.getJobs().get(0);
        return null;
    }

    public JobStatus acknowledgeJob(final Job job) {

        LOGGER.info("AcknowledgeJob for job '{}' and nonce '{}'", job.getId(), job.getNonce());
        final AcknowledgeJobRequest request = new AcknowledgeJobRequest();
        request.setJobId(job.getId());
        request.setNonce(job.getNonce());
        final AcknowledgeJobResult result = codePipelineClient.acknowledgeJob(request);
        return JobStatus.valueOf(result.getStatus());
    }

    public void sendJobSuccessStatus(final Job job, final String message) {
        LOGGER.info("PutJobSuccessResult for job '{}'", job.getId());
        final PutJobSuccessResultRequest request = new PutJobSuccessResultRequest();
        request.setJobId(job.getId());

        ExecutionDetails executionDetails = new ExecutionDetails();
        executionDetails.setSummary(message);

        request.setExecutionDetails(executionDetails);
        codePipelineClient.putJobSuccessResult(request);
    }

    public boolean hasArtifact(final Job job) {
        return !job.getData().getInputArtifacts().isEmpty();
    }

    public Map<String, String> getJobData(final Job job) {
        return job.getData().getActionConfiguration().getConfiguration();
    }

    public String getJobData(final Job job, final String key) {
        return this.getJobData(job).get(key);
    }

    public Path getPackagePath(final Job job) {
        return S3Helper.downloadPackageFromJob(job);
    }

    public void sendJobFailureStatus(final Job job, final String message) {
        LOGGER.info("PutJobFailureResult for job '{}'", job.getId());
        final PutJobFailureResultRequest request = new PutJobFailureResultRequest();
        request.setJobId(job.getId());

        FailureDetails failureDetails = new FailureDetails();
        failureDetails.setType(FailureType.JobFailed);
        failureDetails.setMessage(message);
        request.setFailureDetails(failureDetails);
        codePipelineClient.putJobFailureResult(request);
    }

    String getVersion() {
        if (version != null)
            return version;

        ListActionTypesRequest request = new ListActionTypesRequest();
        ListActionTypesResult actionTypes = codePipelineClient.listActionTypes(request);
        List<String> versionList = new ArrayList<>();
        for (ActionType action : actionTypes.getActionTypes()) {
            ActionTypeId id = action.getId();
            if (id.getCategory().equals(Constants.CODEPIPELINE_ACTION_CATEGORY) && id.getProvider().equals(Constants.CODEPIPELINE_ACTION_PROVIDER_NAME)) {
                versionList.add(id.getVersion());
            }
        }
        validateVersion(versionList);
        version = versionList.get(0);
        return version;
    }

    private void validateVersion(List<String> versions) {
        if (versions.size() == 0)
            throw new CodePipelinePluginExecutionException("XL Deploy action is unavailable on CodePipeline");
        if (versions.size() > 1)
            throw new CodePipelinePluginExecutionException(String.format("Multiple versions %s of XL Deploy actions are configured on CodePipeline, only one version is expected", versions));
    }

    private ActionTypeId getActionTypeId() {

        ActionTypeId actionTypeId = new ActionTypeId();
        actionTypeId.setCategory(ActionCategory.Deploy);
        actionTypeId.setOwner(ActionOwner.Custom);
        actionTypeId.setProvider(Constants.CODEPIPELINE_ACTION_PROVIDER_NAME);
        actionTypeId.setVersion(getVersion());
        return actionTypeId;
    }
}
