import mimetypes
import tempfile
from azure.storage.blob import BlockBlobService, ContentSettings
from xld.azure import string_or_none
from com.xebialabs.deployit.plugin.azure import LoaderUtil
from java.nio.file import Files, Paths, StandardCopyOption


class BlobFileHelper():
    def __init__(self, deployed, container):
        self.deployed = deployed
        self.container = container
        verify_ssl = container.verifySSL
        if not verify_ssl:
            self._disable_https_warnings()
            self._set_ca_bundle_path()

    def __get_block_blobservice_from_deployed(self, the_deployed, container):
        blobService = BlockBlobService(
            account_name=the_deployed.storageAccountName,
            account_key=the_deployed.accountKey,
            sas_token=string_or_none(the_deployed.sasToken),
            protocol=string_or_none(the_deployed.protocol),
            custom_domain=string_or_none(the_deployed.customDomain)
        )

        if container.proxyHost and container.proxyProtocol and container.proxyPort:
            if container.proxyUser and container.proxyPassword:
                blobService.set_proxy(container.proxyHost, container.proxyPort, user=container.proxyUser, password=container.proxyPassword)
            else:
                blobService.set_proxy(container.proxyHost, container.proxyPort)

        return blobService

    def get_block_blobservice_from_deployed(self):
        return self.__get_block_blobservice_from_deployed(self.deployed, self.container)

    def upload_file_into_container(self, container_name, file_name, file_path):
        print file_path
        content_settings = ContentSettings(content_type=mimetypes.guess_type(file_path)[0])
        result = self.upload_blob(
            container_name=container_name,
            file_name=file_name,
            uploaded_file=file_path,
            content_settings=content_settings
        )
        return result

    def delete_file_from_container(self, container_name, file_name):
        return self.delete_blob(container_name, file_name)

    def upload_blob(self, container_name, file_name, uploaded_file, content_settings=None):
        block_blob_service = self.__get_block_blobservice_from_deployed(self.deployed.container, self.container)

        # workaround over jython chunk upload
        BlockBlobService.MAX_SINGLE_PUT_SIZE = 64 * 1024
        BlockBlobService.MAX_BLOCK_SIZE = 45 * 1024
        BlockBlobService.MIN_LARGE_BLOCK_UPLOAD_THRESHOLD = 45 * 1024 + 1

        result = block_blob_service.create_blob_from_path(
            container_name=container_name,
            blob_name=file_name,
            file_path=uploaded_file,
            content_settings=content_settings,
            validate_content=True,
            max_connections=1
        )
        return result

    def delete_blob(self, container_name, file_name):
        block_blob_service = self.__get_block_blobservice_from_deployed(self.deployed.container, self.container)
        result = block_blob_service.delete_blob(
            container_name=container_name,
            blob_name=file_name
        )
        return result

    def _disable_https_warnings(self):
        import requests.packages.urllib3
        requests.packages.urllib3.disable_warnings()

    def _set_ca_bundle_path(self):
        ca_bundle_path = self._extract_file_from_jar("certifi/cacert.pem")
        import os
        os.environ['REQUESTS_CA_BUNDLE'] = ca_bundle_path

    def _extract_file_from_jar(self, config_file):
        file_url = LoaderUtil.getResourceBySelfClassLoader(config_file)
        if file_url:
            tmp_file, tmp_abs_path = tempfile.mkstemp()
            tmp_file.close()
            Files.copy(file_url.openStream(), Paths.get(tmp_abs_path), StandardCopyOption.REPLACE_EXISTING)
            return tmp_abs_path
        else:
            return None
