package com.xebialabs.deployit.plugin.cloud.vsphere.access;

import java.io.Serializable;
import java.net.MalformedURLException;
import java.net.URL;
import java.rmi.RemoteException;
import java.util.Arrays;
import java.util.List;
import com.google.common.base.Joiner;
import com.vmware.vim25.ManagedObjectReference;
import com.vmware.vim25.mo.*;


import com.xebialabs.deployit.plugin.cloud.vsphere.ci.Credentials;

import static com.xebialabs.deployit.plugin.cloud.vsphere.access.VsphereUtils.hostPath;
import static com.xebialabs.deployit.plugin.cloud.vsphere.access.VsphereUtils.vmPath;

public class VsphereAdapter implements Serializable {

    /**
     * Do not use this variable directly, it might be not initialized. Use {@link #getSi()} instead.
     */
    private transient ServiceInstance si;

    private Credentials credentials;

    public VsphereAdapter(final Credentials credentials) {
        this.credentials = credentials;
    }

    public void logout() {
        if (si != null) {
            si.getServerConnection().logout();
        }
    }

    /**
     * Ensures that si is available and returns it
     */
    public ServiceInstance getSi() {
        if (si == null) {
            try {
                si = new ServiceInstance(new URL(credentials.getUrl()), credentials.getUsername(), credentials.getPassword(), credentials.isIgnoreCert());
            } catch (MalformedURLException e) {
                throw new IllegalArgumentException(e);
            } catch (RemoteException e) {
                throw new VsphereAdapterException(e.getMessage());
            }
        }
        return si;
    }

    public Datacenter getDatacenter(String name)  {
        try {
            return (Datacenter) getSi().getSearchIndex().findByInventoryPath(name);
        } catch (RemoteException e) {
            throw new VsphereAdapterException("Can not find datacenter by name " + name + " : " + e.getMessage());
        }
    }

    public VirtualMachine getVm(String datacenter, String path) {
        String p = vmPath(datacenter, path);
        return getVm(p);
    }

    public VirtualMachine getVm(final String path) {
        try {
            return (VirtualMachine) getSi().getSearchIndex().findByInventoryPath(path);
        } catch (RemoteException e) {
            throw new VsphereAdapterException("Can not get template " + path + " : " + e.getMessage());
        }
    }

    public ComputeResource getHost(String datacenter, String path) {
        String p = hostPath(datacenter, path);
        try {
            return (ComputeResource) getSi().getSearchIndex().findByInventoryPath(p);
        } catch (RemoteException e) {
            throw new VsphereAdapterException("Can not get host " + path + " at datacenter " + datacenter + " : " + e.getMessage());
        }
    }

    public Folder getVmFolder(String datacenter, String path) {
        String p = vmPath(datacenter, path);
        try {

            ManagedEntity target = getSi().getSearchIndex().findByInventoryPath(p);
            Folder folder = (target instanceof Datacenter) ? ((Datacenter) target).getVmFolder() : (Folder) target;

            List<String> allowedTypes = Arrays.asList(folder.getChildType());

            if (!allowedTypes.contains("VirtualMachine")) {
                throw new IllegalArgumentException(
                        path + " child types do not allow VirtualMachine type. Allowed types: " + Joiner.on(", ").join(allowedTypes)
                );
            }

            return folder;
        } catch (RemoteException e) {
            throw new VsphereAdapterException("Can not get folder " + path + " at datacenter " + datacenter + " : " + e.getMessage());
        }
    }

    public Task getTask(String morId) {
        ManagedObjectReference mor = new ManagedObjectReference();
        mor.setType("Task");
        mor.setVal(morId);
        return new Task(getSi().getServerConnection(), mor);
    }

    public String getIpAddress(String path) {
        return (String)getVm(path).getPropertyByPath("guest.ipAddress");
    }



    public static class VsphereAdapterException extends RuntimeException {
        public VsphereAdapterException(final String message) {
            super(message);
        }
    }
}
