package com.xebialabs.deployit.plugin.cloud.step;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Set;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.cloud.ci.Environment;
import com.xebialabs.deployit.plugin.cloud.ci.EnvironmentTemplate;
import com.xebialabs.deployit.plugin.cloud.step.rule.SingleEnvCheck;
import com.xebialabs.deployit.plugin.cloud.util.CiFilters;
import com.xebialabs.deployit.plugin.cloud.util.CiParser;
import com.xebialabs.deployit.plugin.cloud.util.ContextHelper;
import com.xebialabs.deployit.plugin.cloud.util.InstanceDescriptorResolver;
import com.xebialabs.deployit.plugin.overthere.Host;

import freemarker.template.TemplateException;

import static com.google.common.base.Predicates.not;
import static com.google.common.collect.Collections2.filter;
import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.collect.Sets.newHashSet;
import static com.google.common.collect.Sets.union;

@SuppressWarnings("serial")
public class RegisterEnvironmentStep implements Step {

    private InstanceDescriptorResolver descriptorResolver;
    private CiParser ciParser;
    private EnvironmentTemplate environmentTemplate;
    private String environmentId;

    public RegisterEnvironmentStep(final InstanceDescriptorResolver descriptorResolver, CiParser ciParser, final EnvironmentTemplate environmentTemplate, final String environmentId) {
        this.descriptorResolver = descriptorResolver;
        this.ciParser = ciParser;
        this.environmentTemplate = environmentTemplate;
        this.environmentId = environmentId;
    }

    @Override
    public int getOrder() {
        return DEFAULT_ORDER + 50;
    }

    @Override
    public String getDescription() {
        return "Register environment " + environmentId;
    }

    @Override
    public StepExitCode execute(final ExecutionContext ctx) throws Exception {
        HashMap<String, Object> scope = newHashMap();

        Set<ConfigurationItem> containers = ContextHelper.wrapped(ctx).safeGet(ContextAttribute.REGISTERED_CONTAINERS.name(),
            Collections.<ConfigurationItem> emptySet());
        Set<Host> hosts = ContextHelper.wrapped(ctx).safeGet(ContextAttribute.REGISTERED_HOSTS.name(), Collections.<Host> emptySet());
        List<ConfigurationItem> parsedCis;

        scope.put("environmentId", environmentId);
        scope.put("containers", containers);
        scope.put("hosts", hosts);
        scope.put("environmentTemplate", environmentTemplate);

        try {
            String descriptor = descriptorResolver.resolve(environmentTemplate.getXmlDescriptor(), scope);
            parsedCis = ciParser.fromString(descriptor, ctx.getRepository());
        } catch (TemplateException e) {
            ctx.logError(e.getMessage());
            return StepExitCode.FAIL;
        }

        try {
            new SingleEnvCheck().check(parsedCis);
        } catch (SingleEnvCheck.Failed e) {
            ctx.logError(e.getMessage());
            return StepExitCode.FAIL;
        }


        Environment cloudEnvironment = (Environment) filter(parsedCis, CiFilters.CloudEnvironments).iterator().next();

        Collection<ConfigurationItem> otherCis = filter(parsedCis, not(CiFilters.CloudEnvironments));

        Collection<ConfigurationItem> topLevelCloudContainers = filter(containers, CiFilters.CloudContainers);

        cloudEnvironment.setLinkedCis(union(newHashSet(otherCis), newHashSet(topLevelCloudContainers)));
        cloudEnvironment.setTemplate(environmentTemplate);

        ctx.getRepository().create(parsedCis.toArray(new ConfigurationItem[parsedCis.size()]));
        return StepExitCode.SUCCESS;
    }


}