package com.xebialabs.deployit.plugin.cloud.ci;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeoutException;

import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.ControlTask;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.deployit.plugin.cloud.step.RegisterInstancesStep;
import com.xebialabs.deployit.plugin.cloud.step.ValidateHostDescriptorStep;
import com.xebialabs.deployit.plugin.cloud.util.CiParser;
import com.xebialabs.deployit.plugin.cloud.util.InstanceDescriptorResolver;
import com.xebialabs.deployit.plugin.cloud.validation.MarkerFileValidation;
import com.xebialabs.overthere.OperatingSystemFamily;

import static com.google.common.collect.Lists.newArrayList;

@SuppressWarnings("serial")
@Metadata(description = "Base class for instance templates, all instance templates must extend it")
@MarkerFileValidation
public abstract class BaseHostTemplate extends BaseConfigurationItem {

    @Property(required = true, size = Property.Size.LARGE, label = "Instance descriptor", description = "Freemarker template of XML which describes instance and middleware")
    private String xmlDescriptor;

    // Connection

    @Property(required = false, category = "Marker", label = "OS family")
    private OperatingSystemFamily os;

    @Property(required = false, category = "Marker", label = "Connection type", description = "Connection type to be used for connecting to the host")
    private ConnectionType connectionType;

    @Property(required = false, category = "Marker", label = "Username")
    private String username;

    @Property(required = false, category = "Marker", label = "Password", password = true)
    private String password;

    @Property(required = false, category = "Marker", label = "Private key file to use for authentication")
    private String privateKeyFile;

    @Property(required = false, category = "Marker", label = "Path to the marker file", description = "Path to the file which should appear on the instance when provisioning completes.")
    private String markerPath;

    @Property(required = true, defaultValue = "500", label = "Boot timeout", description = "Maximal amount of time (in seconds) allowed to elapse before the instance is ready.")
    private int bootTimeout;

    @Property(required = false, defaultValue = "5", category = "Marker", label = "Retry delay", description = "Delay (in seconds) after each connection attempt.")
    private int retryDelay;

    @ControlTask(label = "Validate descriptor", description = "Validate XML descriptor")
    public List<? extends Step> validateDescriptor() {
        return newArrayList(new ValidateHostDescriptorStep(getXmlDescriptor(), this));
    }

    @ControlTask(label = "Instantiate", description = "Create instance from template", parameterType = "cloud.HostParameters")
    public List<? extends Step> instantiate(HostParameters params) {
        ArrayList<Step> steps = newArrayList(produceCreateSteps(params.getInstanceName(), 0));
        steps.add(new RegisterInstancesStep(new InstanceDescriptorResolver(), new CiParser(), params.getHostsLocation()));
        return steps;
    }

    public String getXmlDescriptor() {
        return xmlDescriptor;
    }

    public void setXmlDescriptor(final String xmlDescriptor) {
        this.xmlDescriptor = xmlDescriptor;
    }

    public void setPrivateKeyFile(final String privateKeyFile) {
        this.privateKeyFile = privateKeyFile;
    }

    public void setMarkerPath(final String markerPath) {
        this.markerPath = markerPath;
    }

    public String getPrivateKeyFile() {
        return privateKeyFile;
    }

    public OperatingSystemFamily getOs() {
        return os;
    }

    public void setOs(final OperatingSystemFamily os) {
        this.os = os;
    }

    public String getUsername() {
        return username;
    }

    public String getPassword() {
        return password;
    }

    public String getMarkerPath() {
        return markerPath;
    }

    public int getBootTimeout() {
        return bootTimeout;
    }

    public int getRetryDelay() {
        return retryDelay;
    }

    public ConnectionType getConnectionType() {
        return connectionType;
    }

    public void setConnectionType(final ConnectionType connectionType) {
        this.connectionType = connectionType;
    }

    /**
     * Implementations should provide list of steps needed to create instance from template
     */
    public abstract List<? extends Step> produceCreateSteps(String name, int instanceSeq);

    /**
     * Implementations should provide list of steps needed to destroy instance
     */
    public abstract List<? extends Step> produceDestroySteps(ConfigurationItem instance);

    /**
     * Should return instance public IP address *and* wait until instance is considered running according to hypervisor
     */
    public abstract String getInstanceIpAddress(String cloudId) throws TimeoutException;
}
