package com.xebialabs.deployit.plugin.ec2.access;

import java.io.Serializable;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.TimeoutException;
import org.jclouds.ContextBuilder;
import org.jclouds.aws.ec2.AWSEC2Api;
import org.jclouds.aws.ec2.options.AWSRunInstancesOptions;
import org.jclouds.ec2.EC2Api;
import org.jclouds.ec2.domain.InstanceState;
import org.jclouds.ec2.domain.Reservation;
import org.jclouds.ec2.domain.RunningInstance;
import org.jclouds.ec2.features.InstanceApi;
import org.jclouds.ec2.features.TagApi;
import com.google.common.base.Optional;
import com.google.common.collect.Maps;

import com.xebialabs.deployit.plugin.cloud.util.Retrier;
import com.xebialabs.deployit.plugin.ec2.ci.Credentials;
import com.xebialabs.deployit.plugin.ec2.ci.HostTemplate;

import static com.google.common.base.Objects.firstNonNull;
import static com.google.common.base.Strings.isNullOrEmpty;
import static com.google.common.collect.Iterables.getOnlyElement;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;

@SuppressWarnings("serial")
public class EC2Adapter implements Serializable {

    protected Credentials credentials;

    private int actionTimeout = 20; // seconds

    private int retryDelay = 5; // seconds

    public EC2Adapter(final Credentials credentials) {
        this.credentials = credentials;
    }


    /**
     * Alias with default timing parameters
     */
    public String kickNewInstance(final HostTemplate nodeTemplate, final String instanceLabel) throws TimeoutException {
        return kickNewInstance(nodeTemplate, instanceLabel, actionTimeout, retryDelay, Collections.<String, String>emptyMap());
    }

    /**
     * Triggers new instance creation and gives back an ID.
     * Instance is not ready at this point yet.
     */
    public String kickNewInstance(final HostTemplate nodeTemplate, String instanceLabel, int actionTimeout, int retryDelay, Map<String,String> tags) throws TimeoutException {

        final AWSEC2Api client = client(credentials);
        AWSRunInstancesOptions instanceOptions = AWSRunInstancesOptions.Builder.
                asType(nodeTemplate.getInstanceType().getValue())
                .withSecurityGroupIds(nodeTemplate.getGroups())
                .withKeyName(nodeTemplate.getKeyPair());

        if (!isNullOrEmpty(nodeTemplate.getVpcSubnetId())) {
            instanceOptions.withSubnetId(nodeTemplate.getVpcSubnetId());
        }

        if (!isNullOrEmpty(nodeTemplate.getIamInstanceProfileArn())) {
            instanceOptions.withIAMInstanceProfileArn(
                nodeTemplate.getIamInstanceProfileArn());
        }

        Reservation<? extends RunningInstance> started = client.getInstanceApi().get().
                runInstancesInRegion(nodeTemplate.getRegion(), null, nodeTemplate.getAmi(), 1, 1, instanceOptions);

        final RunningInstance instance = getOnlyElement(started);

        final Map<String, String> instanceTags = newHashMap(firstNonNull(tags, Maps.<String, String>newHashMap()));
        instanceTags.put("Name", instanceLabel);

        new Retrier<Boolean>(
                new Callable<Boolean>() {
                    @Override
                    public Boolean call() throws Exception {
                        getTagApi(client, nodeTemplate).applyToResources(instanceTags, newArrayList(instance.getId()));
                        return true;
                    }
                }
        ).retryFor(actionTimeout, retryDelay);

        return instance.getId();
    }

    /**
     * Waits until instance becomes available according to the information retrieved from hypervisor
     * @return running instance object
     */
    public RunningInstance waitForRunningInstance(final String region, final String instanceId, final int actionTimeout, final int retryDelay) throws TimeoutException {
        final EC2Api client = client(credentials);

        return new Retrier<RunningInstance>(new Callable<RunningInstance>() {
            @Override
            public RunningInstance call() throws Exception {
                RunningInstance instance = getOnlyElement(getOnlyElement(getInstanceApi(client).describeInstancesInRegion(region, instanceId)));
                if (InstanceState.RUNNING.equals(instance.getInstanceState())) {
                    return instance;
                }

                throw new IllegalStateException("Expected state to be RUNNING, got " + instance.getInstanceState() + " instead");
            }
        }).retryFor(actionTimeout, retryDelay);
    }

    /**
     * Returns IP address of the instance
     * @return Public address of the running instance
     */
    public String getPublicIpAddress(final String region, final String instanceId) {
        return getOnlyElement(getOnlyElement(client(credentials).getInstanceApi().get().describeInstancesInRegion(region, instanceId))).getIpAddress();
    }

    /**
     * Destroys the instance
     */
    public void shutDown(final String region, final String cloudId) {
        client(credentials).getInstanceApi().get().terminateInstancesInRegion(region, cloudId);
    }


    // Helper


    private AWSEC2Api client(Credentials credentials) {
        return ContextBuilder.newBuilder("aws-ec2").
                credentials(credentials.getKey(), credentials.getSecret()).
                buildApi(AWSEC2Api.class);
    }

    private TagApi getTagApi(final AWSEC2Api client, final HostTemplate nodeTemplate) {
        final Optional<? extends TagApi> tagApiForRegionOptional = client.getTagApiForRegion(nodeTemplate.getRegion());
        if(tagApiForRegionOptional.isPresent()){
            return tagApiForRegionOptional.get();
        }
        throw new IllegalStateException("Could not find getTagApiForRegion for " + EC2Api.class);
    }

    private InstanceApi getInstanceApi(final EC2Api client) {
        final Optional<? extends InstanceApi> instanceApiOptional = client.getInstanceApi();
        if(instanceApiOptional.isPresent()){
            return instanceApiOptional.get();
        }
        throw new IllegalStateException("Could not find getInstanceApi for " + EC2Api.class);
    }
}
