package com.xebialabs.deployit.plugin.wps.contributor;

import java.util.Map;
import com.google.common.collect.Iterables;

import com.xebialabs.deployit.plugin.api.deployment.planning.Contributor;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.Deltas;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication;
import com.xebialabs.deployit.plugin.python.PythonDeploymentStep;
import com.xebialabs.deployit.plugin.was.deployed.ExtensibleDeployed;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Maps.newHashMap;
import static com.xebialabs.deployit.plugin.api.deployment.specification.Operation.DESTROY;
import static com.xebialabs.deployit.plugin.api.deployment.specification.Operation.MODIFY;

public class BpcStopForceDeleteProcesses {

    @Contributor
    public void applyBpcStopForceDeleteProcesses(Deltas deltas, DeploymentPlanningContext ctx) {
        Iterable<Delta> wpsEarDeltas = findWpsEarDeltas(deltas);
        Boolean bpcForceUndeployment = ctx.getDeployedApplication().getProperty("bpcForceUndeployment");

        for (Delta wpsEarDelta : wpsEarDeltas) {
            Operation operation = wpsEarDelta.getOperation();
            if (operation == MODIFY || operation == DESTROY ) {
                ExtensibleDeployed<?> wpsEarModule = extractDeployed(wpsEarDelta);
                Boolean stopProcessAndTaskTemplates = wpsEarModule.getProperty("stopProcessAndTaskTemplates");
                Boolean deleteInstancesAndStopTemplates = wpsEarModule.getProperty("deleteInstancesAndStopTemplates");
                if (bpcForceUndeployment || deleteInstancesAndStopTemplates) {
                    Integer order = wpsEarModule.getProperty("deleteBpcTemplatesScriptOrder");
                    String script = wpsEarModule.getProperty("deleteBpcTemplatesScript");
                    addStep(order, script, "Deleting processes and tasks for " + wpsEarModule.getName(), wpsEarModule, ctx);
                } else if (stopProcessAndTaskTemplates) {
                    Integer order = wpsEarModule.getProperty("stopBpcTemplatesScriptOrder");
                    String script = wpsEarModule.getProperty("stopBpcTemplatesScript");
                    addStep(order, script, "Stopping processes and tasks for " + wpsEarModule.getName() , wpsEarModule, ctx);
                }
            }
        }
    }

    private void addStep(Integer order, String script, String description, ExtensibleDeployed<?> deployed, DeploymentPlanningContext ctx) {
        PythonDeploymentStep step = new PythonDeploymentStep(
                order,
                deployed.getContainer().getManagingContainer(),
                script,
                getPythonVars(ctx.getDeployedApplication(), deployed),
                description);
        ctx.addStep(step);
    }

    private ExtensibleDeployed<?> extractDeployed(Delta delta) {
        if (delta.getDeployed() == null) {
            return (ExtensibleDeployed<?>) delta.getPrevious();
        }
        return (ExtensibleDeployed<?>) delta.getDeployed();
    }

    private Map<String, Object> getPythonVars(DeployedApplication deployedApplication, ExtensibleDeployed<?> wpsEarModule) {
        Map<String, Object> pythonVars = newHashMap();
        pythonVars.put("deployed", wpsEarModule);
        Boolean exposeDeployedApplication = wpsEarModule.getProperty("exposeDeployedApplication");
        if(exposeDeployedApplication) {
            checkNotNull(deployedApplication, "deployedApplication is null");
            pythonVars.put("deployedApplication", deployedApplication);
        }
        return pythonVars;
    }

    private Iterable<Delta> findWpsEarDeltas(Deltas deltas) {
        return Iterables.filter(deltas.getDeltas(), new WpsEarModulePredicate());
    }
}