#
import re
import xml.etree.ElementTree as ET
from xml.dom import minidom
from wlp.modules import Struct
from wlp.modules.utility import Logger


class DeployedParser(object):

    def __init__(self):
        pass

    def parse(self, deployed):
        element = Struct()
        self._parse_configuration_properties(deployed, element, prefix="", propertyMapping='propertyMapping')
        self._parse_configuration_properties(deployed, element, prefix="", propertyMapping='additionalPropertyMapping')
        self._parse_additional_properties(deployed, element)
        return element

    def _parse_configuration_properties(self, deployed, element, prefix="", propertyMapping='propertyMapping'):
        """
        Parses deployed.propertyMapping and returns a Struct representation of server.xml element.
        Resulting Struct element represent structure of the XML element.

        Example input:
            - deployed.transactional = 'true'
            - deployed.sybaseDatabaseName = 'someDb'
            - deployed.propertyMapping = {'transactional':, 'properties.sybase/databaseName':'sybaseDatabaseName'}
        Example output:
            - element (Struct) = {'transactional':'true', 'properties.sybase':{'databaseName':'someDb'}}
        """
        if hasattr(deployed, propertyMapping):
            for attr_name, prop_name in getattr(deployed, propertyMapping).iteritems():
                if attr_name and (not prefix or attr_name.startswith(prefix)):
                    nested_attrs = attr_name[len(prefix):].split("/")
                    current_attr = nested_attrs[0]
                    if current_attr in element:
                        continue
                    prop_name = prop_name if prop_name else attr_name
                    attr_value = self._get_value(deployed, prop_name) if len(nested_attrs) == 1 else self._parse_configuration_properties(deployed, element=Struct(), prefix="%s%s/" % (prefix, current_attr), propertyMapping=propertyMapping)
                    if attr_value:
                        element[str(current_attr)] = attr_value
        return element

    def _parse_additional_properties(self, deployed, element):
        """
        Parses deployed.customProperties (map_string_string) as attributes of deployed.customPropertiesElementName Struct() entry of provided element.

        Keyword args:
            - deployed - deployed to parse
            - element  - struct into which parsed data will be placed

        Example input:
            - deployed.customPropertiesElementName = 'properties.db2'
            - deployed.customProperties = 'driverType:4'
        Example output:
            - element (Struct) - {'properties.db2': {'driverType':'4'}}
        """
        if hasattr(deployed, 'customProperties') and hasattr(deployed, 'customPropertiesElementName'):
            element_name = str(deployed.customPropertiesElementName)
            s = getattr(element, element_name, Struct())
            for prop_name, prop_val in deployed.customProperties.iteritems():
                s[str(prop_name)] = str(prop_val)
        return element


    def _get_value(self, deployed, property_name):
        deployed_property_name = property_name.replace("/", "_")
        deployed_property_value = getattr(deployed, deployed_property_name)
        return self._convert_value(deployed_property_value)

    def _convert_value(self, value):
        if isinstance(value, bool):
            return value and 'true' or 'false'
        elif hasattr(value, '__iter__'):
            # TODO: see if support for LIST_OF_STRING etc is required.
            if len(value) > 0:
                return [self.parse(embedded) for embedded in value]
        elif value is not None:
            return str(value)


class XmlFormatter(object):
    DEFAULT_ENCODING = 'utf-8'

    def format(self, elem):
        return ET.tostring(elem, self.DEFAULT_ENCODING)

    def parse(self, content):
        return ET.fromstring(content.encode(self.DEFAULT_ENCODING, 'ignore'))


class PrettyXmlFormatter(XmlFormatter):
    re_space = re.compile('>\n\s+([^<>\s].*?)\n\s+</', re.DOTALL)

    def format(self, elem):
        not_pretty_xml = ET.tostring(elem, self.DEFAULT_ENCODING)
        document = minidom.parseString(not_pretty_xml)
        nearly_pretty_xml = document.toprettyxml(indent="  ")
        pretty_xml = self.re_space.sub('>\g<1></', nearly_pretty_xml)
        return pretty_xml


class Configuration(object):
    """
    """
    def __init__(self, content, parser=DeployedParser(), formatter=PrettyXmlFormatter()):
        self.parser = parser
        self.formatter = formatter
        try:
            self.server_root = formatter.parse(content)
        except ET.ParseError:
            Logger.log_and_raise_error("Invalid format of server.xml, unable to parse.")

    def create_or_update_resource(self, deployed):
        self.remove_resource(deployed, verbose=False)
        attributes = self.parser.parse(deployed)
        self._create_element(self.server_root, deployed.propertyMappingElementName, attributes)

    def remove_resource(self, deployed, verbose=True):
        resource = self._find_by_id(deployed.propertyMappingElementName, deployed.name)
        if resource is not None:
            self._remove_element(resource)
        elif verbose:
            print "%s with id '%s' already removed" % (deployed.type.name, deployed.name)

    def add_feature(self, deployed):
        feature_manager = self.get_feature_manager()
        feature = self.get_feature(feature_manager, deployed.name)
        if feature is None:
            feature = ET.SubElement(feature_manager, "feature")
            feature.text = deployed.name
        else:
            print "Feature '%s' already installed" % deployed.name

    def remove_feature(self, deployed):
        feature_manager = self.get_feature_manager()
        feature = self.get_feature(feature_manager, deployed.name)
        if feature is None:
            print "Feature '%s' already removed" % deployed.name
        else:
            feature_manager.remove(feature)

    def get_feature_manager(self):
        feature_manager = self.server_root.find("featureManager")
        if feature_manager is None:
            feature_manager = ET.SubElement(self.server_root, "featureManager")
        return feature_manager

    def get_feature(self, feature_manager, name):
        for feature in feature_manager.findall("feature"):
            if feature.text == name:
                return feature

    def to_string(self):
        self._indent(self.server_root)
        return ET.tostring(self.server_root, "utf-8")

    # private

    def _indent(self, elem, level=0):
        i = "\n" + level * "  "
        if len(elem):
            if not elem.text or not elem.text.strip():
                elem.text = i + "  "
            if not elem.tail or not elem.tail.strip():
                elem.tail = i
            for elem in elem:
                self._indent(elem, level + 1)
            if not elem.tail or not elem.tail.strip():
                elem.tail = i
        else:
            if level and (not elem.tail or not elem.tail.strip()):
                elem.tail = i

    def _create_element(self, parent, tag, attributes):
        current_element = ET.SubElement(parent, tag)
        for key, value in attributes.iteritems():
            if isinstance(value, dict):
                self._create_element(current_element, key, value)
            elif isinstance(value, list):
                for nested in value:
                    self._create_element(current_element, key, nested)
            else:
                current_element.set(key, value)
        return current_element

    def _remove_element(self, element):
        self.server_root.remove(element)

    def _find_by_id(self, tag, idd):
        return self._find_by_attr(tag, "id", idd)

    def _find_by_attr(self, tag, attr_name, attr_value):
        for elem in self.server_root.findall(tag):
            if elem.get(attr_name) == str(attr_value):
                return elem