package com.xebialabs.deployit.plugin.was.container;

import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.base.Function;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.inspection.Inspect;
import com.xebialabs.deployit.plugin.api.inspection.InspectionContext;
import com.xebialabs.deployit.plugin.api.inspection.InspectionProperty;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.base.BaseContainer;
import com.xebialabs.deployit.plugin.overthere.Host;
import com.xebialabs.deployit.plugin.python.ControlTaskDelegate;
import com.xebialabs.deployit.plugin.python.PythonManagingContainer;
import com.xebialabs.deployit.plugin.was.step.CheckWasInstallationStep;
import com.xebialabs.deployit.plugin.was.step.DiscoveryWasTopologyStep;
import com.xebialabs.deployit.plugin.was.step.GetWasVersionStep;
import com.xebialabs.overthere.CmdLine;
import com.xebialabs.overthere.OverthereFile;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.plugin.was.container.DelegatedInspectionHelper.discoverDeployeds;

@SuppressWarnings("serial")
@Metadata(description = "Base class for a Webshpere Cell", inspectable = true)
public abstract class BaseCell extends BaseContainer implements Cell {

    @Property(required = true, asContainment = true, description = "Host on which the WAS server runs")
    @InspectionProperty(required = true)
    private Host host;

    @Property(required = true, label = "WebSphere Installation Path", description = "Path to the WebSphere profile or deployment manager profile, where bin/wsadmin can be found. For example, /opt/ws/6.1/appserver/profiles/AppSrv01", size = Property.Size.MEDIUM)
    @InspectionProperty(required = true)
    private String wasHome;

    @Property(required = true, label = "Administrative port", description = "TCP port which is used to login to the WebSphere Administration, defaults to 8880 for SA, 8879 for ND. Value of '0' has special meaning: it'used to suppress passing '-port' argument to wsadmin.")
    @InspectionProperty(required = false)
    private int port;

    @Property(required = false, label = "Administrative username", description = "Username which is used to login to the WebSphere Administration.")
    @InspectionProperty(required = false)
    private String username;

    @Property(required = false, password = true, label = "Administrative password", description = "Password which is used to login to the WebSphere Administration.")
    @InspectionProperty(required = false)
    private String password;

    @Property(required = true, label = "WAS version", description = "Version of WebSphere Application Server.")
    private WasVersion version;

    @Property(required = false, label = "Run with daemon", description = "Set to true to execute commands with the Python daemon", defaultValue = "true", hidden = true)
    private boolean runWithDaemon;

    @Property(required = false, hidden = true, defaultValue = "was/container/discover-inspect-topology-lib.py, was/container/container-lib.py", description = "List of scripts to appended to runtime script")
    private List<String> libraryScripts = newArrayList();

    @Property(hidden = true, defaultValue = "was/container/discover-inspect-topology.py", description="Inspect script for Cell")
    private String inspectScript;

    @Property(hidden = true, defaultValue = "was.VirtualHost,was.SharedLibrary")
    private Set<String> deployedsToDiscover = newHashSet();

    @Property(required=false, label="Hostname", description="Hostname to which wsadmin should attempt to connect.")
    @InspectionProperty(required = false)
    private String hostname;

    private static final String CONFIG_DIR_NAME = "config";

    private static final String PLUGIN_FILENAME = "plugin-cfg.xml";

    @Override
    public int getPort() {
        return port;
    }

    public void setPort(int port) {
        this.port = port;
    }

    @Override
    public String getUsername() {
        return username;
    }

    public void setUsername(String username) {
        this.username = username;
    }

    @Override
    public String getPassword() {
        return password;
    }

    public void setPassword(String password) {
        this.password = password;
    }

    @Override
    public String getWasHome() {
        return wasHome;
    }

    public void setWasHome(String wasHome) {
        this.wasHome = wasHome;
    }

    @Override
    public WasVersion getVersion() {
        return version;
    }

    @Override
    public void setVersion(WasVersion version) {
        this.version = version;
    }

    @Override
    public Cell getCell() {
        return this;
    }

    @Override
    public Host getCellHost() {
        return host;
    }

    @Override
    public Host getHost() {
        return host;
    }

    public void setHost(Host host) {
        this.host = host;
    }

    public String getHostname() {
        return hostname;
    }

    public void setHostname(String hostname) {
        this.hostname = hostname;
    }

    @Override
    public String getConfigDirPath() {
        String fileSep = getCellHost().getOs().getFileSeparator();
        return getWasHome() + fileSep + CONFIG_DIR_NAME;
    }

    @Override
    public String getPluginFilePath() {
        String fileSep = getCellHost().getOs().getFileSeparator();
        return getConfigDirPath() + fileSep + "cells" + fileSep + PLUGIN_FILENAME;
    }

    @Override
    public String getWsadminPath() {
        return getCommandPath("wsadmin");
    }

    @Override
    public String getVersionInfoPath() {
        return getCommandPath("versionInfo");
    }

    @Override
    public String getStartServerPath() {
        return getCommandPath("startServer");
    }

    @Override
    public String getStopServerPath() {
        return getCommandPath("stopServer");
    }

    @Override
    public String getServerStatusPath() {
        return getCommandPath("serverStatus");
    }

    private String getCommandPath(String command) {
        String fileSep = getCellHost().getOs().getFileSeparator();
        String scriptExt = getCellHost().getOs().getScriptExtension();
        return getWasHome() + fileSep + "bin" + fileSep + command + scriptExt;
    }

    @Override
    public boolean runWithDaemon() {
        return runWithDaemon;
    }

    public void setRunWithDaemon(boolean runWithDaemon) {
        this.runWithDaemon = runWithDaemon;
    }

    public List<String> getLibraryScripts() {
        return libraryScripts;
    }

    @Override
    public PythonManagingContainer getManagingContainer() {
        return this;
    }

    @Override
    public CmdLine getScriptCommandLine(OverthereFile pyFile) {
        CmdLine commandLine = new CmdLine();
        commandLine.addArgument(getWsadminPath());
        if (getUsername() != null && !getUsername().trim().isEmpty()) {
            commandLine.addArgument("-user");
            commandLine.addArgument(getUsername());
            if (getPassword() != null && !getPassword().isEmpty()) {
                commandLine.addArgument("-password");
                commandLine.addPassword(getPassword());
            }
        }
        if (getPort() != 0) {
            commandLine.addArgument("-port");
            commandLine.addArgument(Integer.toString(getPort()));
        }
        if (getHostname() != null && !getHostname().trim().isEmpty()) {
            commandLine.addArgument("-host");
            commandLine.addArgument(getHostname());
        }
        commandLine.addArgument("-lang");
        commandLine.addArgument("jython");
        commandLine.addArgument("-f");
        commandLine.addArgument(pyFile.getPath());
        return commandLine;
    }

    public Set<String> getDeployedsToDiscover() {
        return deployedsToDiscover;
    }

    public void setDeployedsToDiscover(Set<String> deployedsToDiscover) {
        this.deployedsToDiscover = deployedsToDiscover;
    }

    @Override
    public String getRuntimePath() {
        return "was/runtime";
    }

    public String getInspectScript() {
        return inspectScript;
    }

    public List<Step> controlTaskDispatch(String name, Map<String, String> args) {
        return ControlTaskDelegate.dispatch(name, args, this, this);
    }

    @Inspect
    public void inspect(InspectionContext ctx) {
        if(!skipTopologyDiscovery) {
            ctx.addStep(new CheckWasInstallationStep(this));
            ctx.addStep(new GetWasVersionStep(this));
        }

        ctx.addStep(new DiscoveryWasTopologyStep(this,skipTopologyDiscovery));

        if (!performOnlyTopologyDiscovery) {
            Iterable<Type> typesToDiscovery = Iterables.transform(deployedsToDiscover, new Function<String, Type>() {
                @Override
                public Type apply(String input) {
                    return Type.valueOf(input);
                }
            });
            discoverDeployeds(this, ctx, Lists.newArrayList(typesToDiscovery));
        }
    }

    // TODO: Remove when the ability to skip discovery steps is available
    public boolean skipTopologyDiscovery;
    public boolean performOnlyTopologyDiscovery;

}
