package com.xebialabs.deployit.plugin.was.step;

import com.google.common.io.Closeables;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.deployit.plugin.overthere.Host;
import com.xebialabs.deployit.plugin.was.container.Cell;
import com.xebialabs.overthere.OverthereConnection;
import com.xebialabs.overthere.OverthereFile;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.xebialabs.overthere.util.OverthereFileCopier.copy;
import static java.lang.String.format;

@SuppressWarnings("serial")
public class GlobalWebServerPluginConfigurationCopyStep implements Step {
    private final int order;
    private final String description;
    private final Host srcHost;
    private final Host targetHost;
    private final String pluginSrcFilePath;
    private final String pluginDestFilePath;

    public GlobalWebServerPluginConfigurationCopyStep(int order, Cell cell, BaseConfigurationItem unmanagedWebserver) {
        this.order = order;
        srcHost = cell.getCellHost();
        targetHost = unmanagedWebserver.getProperty("host");
        pluginSrcFilePath = format("%s/config/cells/plugin-cfg.xml", cell.getWasHome());
        pluginDestFilePath = checkNotNull(unmanagedWebserver.<String>getProperty("pluginConfigurationPath"), "pluginConfigurationPath");
        description = format("Copy global web server plugin configuration file directly to '%s' for webserver '%s'",
                targetHost.getName(), unmanagedWebserver.getName());
    }

    @Override
    public String getDescription() {
        return description;
    }

    @Override
    public int getOrder() {
        return order;
    }

    @Override
    public StepExitCode execute(ExecutionContext ctx) throws Exception {
        OverthereConnection srcHostConnection = srcHost.getConnection();
        OverthereConnection targetHostConnection = targetHost.getConnection();
        try {
            final OverthereFile source = srcHostConnection.getFile(pluginSrcFilePath);
            final OverthereFile remote = targetHostConnection.getFile(pluginDestFilePath);
            ctx.logOutput(format("Copying generated plugin configuration file from '%s' on '%s' to '%s' on '%s'",
                    pluginSrcFilePath, srcHost, pluginDestFilePath, targetHost));
            copy(source, remote);
        } finally {
            Closeables.close(srcHostConnection, true);
            Closeables.close(targetHostConnection, true);
        }
        return StepExitCode.SUCCESS;
    }
}
