package com.xebialabs.deployit.plugin.was.deployed.datasource;

import com.google.common.collect.ImmutableMap;

import com.xebialabs.deployit.plugin.api.deployment.planning.Contributor;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.specification.Deltas;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.python.PythonDeploymentStep;
import com.xebialabs.deployit.plugin.was.contributor.CollectDeployedsOfTypesContributor;
import com.xebialabs.deployit.plugin.was.deployed.ExtensibleDeployedResource;

import static java.lang.String.format;
import static com.google.common.collect.Iterables.concat;
import static com.google.common.collect.Iterables.transform;
import static com.xebialabs.deployit.plugin.was.util.Predicates.extractDeployed;
import static java.lang.Boolean.TRUE;

/**
 * Contributor to add test data source step. Before the data source can be tested any changed JAAS aliases have to be
 * updated. This is done by another contributor, which must add steps before the ones added by this one.
 */
public class TestDatasourceContributor extends CollectDeployedsOfTypesContributor<ExtensibleDeployedResource> {
    private static final Type DATASOURCE_TYPE = Type.valueOf("was.Datasource");
    private static final String DATASOURCE_TEST_CONNECTION_PROPERTY = "testConnection";

    private static final String TEST_CONNECTION_SCRIPT_PROPERTY = "testConnectionScript";
    private static final String TEST_CONNECTION_ORDER_PROPERTY = "testConnectionOrder";

    public TestDatasourceContributor() {
        super(DATASOURCE_TYPE);
    }

    @Contributor
    public void addTestConnectionSteps(Deltas deltas, DeploymentPlanningContext ctx) {
        filterDeltas(deltas.getDeltas());
        addTestConnectionStep(ctx);
    }

    private void addTestConnectionStep(DeploymentPlanningContext ctx) {
        Iterable<Deployed<?, ?>> createdOrModifiedDatasources = concat(
            deployedsCreated,
            transform(deployedsModified, extractDeployed())
        );
        for (Deployed<?, ?> datasource : createdOrModifiedDatasources) {
            if (TRUE.equals(datasource.getProperty(DATASOURCE_TEST_CONNECTION_PROPERTY))) {
                addTestConnectionStep((ExtensibleDeployedResource) datasource, ctx);
            }
        }
    }

    public static void addTestConnectionStep(ExtensibleDeployedResource datasource, DeploymentPlanningContext ctx) {
        ctx.addStep(new PythonDeploymentStep(datasource.<Integer> getProperty(TEST_CONNECTION_ORDER_PROPERTY),
            datasource.getContainer().getCell(),
            datasource.<String> getProperty(TEST_CONNECTION_SCRIPT_PROPERTY),
            ImmutableMap.<String, Object> of("deployed", datasource),
            format("Test datasource connection to '%s'", datasource.getName()))
        );
    }
}
