package com.xebialabs.deployit.plugin.was.contributor;

import com.google.common.collect.ImmutableMap;

import com.xebialabs.deployit.plugin.api.deployment.planning.Contributor;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.specification.Deltas;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.python.PythonDeploymentStep;
import com.xebialabs.deployit.plugin.was.deployed.ExtensibleDeployedResource;

import static com.google.common.collect.Iterables.concat;
import static com.google.common.collect.Iterables.transform;
import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.plugin.was.util.Predicates.extractDeployed;
import static java.lang.String.format;

/** Contributor that adds an update JaasAlias step that pushes changed JaasAlias data to runtime. */
public class UpdateJaasAliasContributor extends CollectDeployedsOfTypesContributor<ExtensibleDeployedResource> {
    private static final Type DATASOURCE_TYPE = Type.valueOf("was.Datasource");
    private static final Type JAAS_ALIAS_TYPE = Type.valueOf("was.JaasAlias");

    private static final String UPDATE_JAAS_SCRIPT_PROPERTY = "updateJaasScript";
    private static final String UPDATE_JAAS_ORDER_PROPERTY = "updateJaasOrder";

    public UpdateJaasAliasContributor() {
        super(newArrayList(DATASOURCE_TYPE, JAAS_ALIAS_TYPE));
    }

    @Contributor
    public void addUpdateStepToPlan(Deltas deltas, DeploymentPlanningContext ctx) {
        filterDeltas(deltas.getDeltas());
        addUpdateJaasAliasStep(ctx);
    }

    private void addUpdateJaasAliasStep(DeploymentPlanningContext ctx) {
        Iterable<Deployed<?, ?>> deployeds = concat(
            deployedsCreated,
            transform(deployedsModified, extractDeployed()),
            deployedsRemoved
        );
        // could limit the amount of update steps per unique cell/script and the max order.
        // a lot of trouble for little gain.
        for (Deployed<?, ?> d : deployeds) {
            addUpdateJaasStep((ExtensibleDeployedResource) d, ctx);
        }
    }

    public static void addUpdateJaasStep(ExtensibleDeployedResource datasource, DeploymentPlanningContext ctx) {
        ctx.addStep(new PythonDeploymentStep(datasource.<Integer> getProperty(UPDATE_JAAS_ORDER_PROPERTY),
            datasource.getContainer().getCell(),
            datasource.<String> getProperty(UPDATE_JAAS_SCRIPT_PROPERTY),
            ImmutableMap.<String, Object> of("deployed", datasource),
            format("Update J2C authentication data at run time"))
        );
    }
}
