package com.xebialabs.deployit.plugin.was.util;

import java.util.Collection;
import java.util.Collections;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.Sets;

import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Deployed;

import static com.xebialabs.deployit.plugin.api.deployment.specification.Operation.DESTROY;
import static com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry.getSubtypes;

public final class Predicates {

    private Predicates()
    {
    }

    public static Predicate<Type> subtypeOf(Type type) {
        return new IsSubtypeOf(type);
    }

    public static Predicate<ConfigurationItem> instanceOf(Type type) {
        return com.google.common.base.Predicates.compose(subtypeOf(type), new Function<ConfigurationItem, Type>() {
            @Override
            public Type apply(ConfigurationItem input) {
                return input.getType();
            }
        });
    }

    public static Predicate<Delta> deltaOf(Type type) {
        return com.google.common.base.Predicates.compose(instanceOf(type), extractDeployed());
    }

    public static Function<Delta, Deployed<?, ?>> extractDeployed() {
        return new ExtractDeployed();
    }

    public static Predicate<Delta> operationIs(Operation operationToMatch) {
        return new OperationEquals(operationToMatch);
    }

    private static class OperationEquals implements Predicate<Delta> {
        private final Operation operationToMatch;

        protected OperationEquals(Operation operationToMatch) {
            this.operationToMatch = operationToMatch;
        }

        @Override
        public boolean apply(Delta input) {
            return input.getOperation().equals(operationToMatch);
        }
    }

    private static class IsSubtypeOf implements Predicate<Type> {
        private final Collection<Type> subtypes;

        public IsSubtypeOf(Type typeToMatch) {
            if (DescriptorRegistry.exists(typeToMatch)) {
                subtypes = Sets.newHashSet(getSubtypes(typeToMatch));
                subtypes.add(typeToMatch);
            } else {
                subtypes = Collections.emptySet();
            }
        }

        @Override
        public boolean apply(Type input) {
            return subtypes.contains(input);
        }
    }

    private static class ExtractDeployed implements Function<Delta, Deployed<?, ?>> {
        @Override
        public Deployed<?, ?> apply(Delta input) {
            return (input.getOperation().equals(DESTROY) ? input.getPrevious() : input.getDeployed());
        }
    }

}
