package com.xebialabs.deployit.plugin.was.container;

import java.util.List;
import java.util.Map;
import java.util.Set;

import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.udm.ControlTask;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.validation.Range;
import com.xebialabs.deployit.plugin.python.ControlTaskDelegate;
import com.xebialabs.deployit.plugin.was.deployable.DynamicClusterSpec;
import com.xebialabs.deployit.plugin.was.util.OperationalMode;

import static com.google.common.collect.Maps.newHashMap;

@SuppressWarnings("serial")
@Metadata(description = "A WebSphere dynamic cluster managed by a WebSphere deployment manager (WAS ND)")
public class DynamicCluster extends BaseCluster<DynamicClusterSpec> {

    @Property(label = "Membership policy", defaultValue = "node_nodegroup = 'DefaultNodeGroup'", description = "Specify an expression that is used to define which nodes are used for the dynamic cluster.Ex:node_nodegroup = 'my_node_group'")
    private String membershipPolicy;

    @Property(required = true, label = "Maximum number of cluster instances", defaultValue = "-1", description = "Choose the Maximum number of instances to have running when the dynamic cluster is activated,The Default is -1(Do not limit the number of instances that can start)")
    @Range(minimum = -1, message = "The Maximum number of instances must be greater than -1")
    private int maxInstances;

    @Property(required = true, label = "Minimum number of cluster instances", defaultValue = "1", description = "Choose the minimum number of instances to have running when the dynamic cluster is activated,The Default is 1")
    @Range(minimum = 0, message = "The minimum number of instances must be a positive value")
    private int minInstances;

    @Property(required = true, label = "Vertical stacking of instances on node", defaultValue = "0", description = "If the nodes in the dynamic cluster have extra capacity, vertical stacking allows an application to make more effective use of the capacity by starting multiple instances on the same node,The Default is 0")
    @Range(minimum = 0, message = "The vertical stacking number of instances must be a positive value.")
    private int numVerticalInstances;

    @Property(required = false, label = "Cluster members", description = "Servers that are part of this cluster", hidden = true)
    private Set<ManagedServer> servers;

    @Property(required = false, defaultValue = "MANUAL", description = "Specifies the level that the dynamic cluster interacts with the dynamic operations environment.")
    private OperationalMode operationalMode;

    @Property(required = false, defaultValue = "was/container/create-dynamic-cluster.py", hidden = true)
    private String createPythonScript;

    @Property(required = false, defaultValue = "was/container/modify-dynamic-cluster.py", hidden = true)
    private String modifyPythonScript;

    @Property(required = false, defaultValue = "was/container/destroy-dynamic-cluster.py", hidden = true)
    private String destroyPythonScript;

    public String getMembershipPolicy() {
        return membershipPolicy;
    }

    public void setMembershipPolicy(String membershipPolicy) {
        this.membershipPolicy = membershipPolicy;
    }

    public int getMinInstances() {
        return minInstances;
    }

    public void setMinInstances(int minInstances) {
        this.minInstances = minInstances;
    }

    public int getMaxInstances() {
        return maxInstances;
    }

    public void setMaxInstances(int maxInstances) {
        this.maxInstances = maxInstances;
    }

    public int getNumVerticalInstances() {
        return numVerticalInstances;
    }

    public void setNumVerticalInstances(int numVerticalInstances) {
        this.numVerticalInstances = numVerticalInstances;
    }

    public OperationalMode getOperationalMode() {
        return operationalMode;
    }

    public void setOperationalMode(OperationalMode operationalMode) {
        this.operationalMode = operationalMode;
    }

    @Override
    public String getContainmentPath() {
        // resource operations (creation of data sources etc.) only works via the "regular" cluster path
        return "/ServerCluster:" + getName();
    }

    @Override
    public String getWasConfigIdType() {
        return "DynamicCluster";
    }

    public String getCreatePythonScript() {
        return createPythonScript;
    }

    public void setCreatePythonScript(String createPythonScript) {
        this.createPythonScript = createPythonScript;
    }

    public String getModifyPythonScript() {
        return modifyPythonScript;
    }

    public void setModifyPythonScript(String modifyPythonScript) {
        this.modifyPythonScript = modifyPythonScript;
    }

    public String getDestroyPythonScript() {
        return destroyPythonScript;
    }

    public void setDestroyPythonScript(String destroyPythonScript) {
        this.destroyPythonScript = destroyPythonScript;
    }

    // Control Tasks
    @ControlTask(label = "Create", description = "This control task is used to create a new DynamicCluster")
    public List<Step> create() {
        return invokeControlTask("create");
    }

    @ControlTask(label = "Modify", description = "This control task is used to modify the properties of an existing DynamicCluster")
    public List<Step> modify() {
        return invokeControlTask("modify");
    }

    @ControlTask(label = "Destroy", description = "This control task is used to destroy an existing DynamicCluster")
    public List<Step> destroy() {
        return invokeControlTask("destroy");
    }

    private List<Step> invokeControlTask(String name) {
        Map<String, String> args = newHashMap();
        return ControlTaskDelegate.dispatch(name, args, this, this);
    }
}
