package com.xebialabs.deployit.plugin.was.contributor.webserver;

import java.util.Set;

import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.ImmutableSet.Builder;

import com.xebialabs.deployit.plugin.api.deployment.planning.Contributor;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.specification.Deltas;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.was.container.Cell;
import com.xebialabs.deployit.plugin.was.contributor.CollectDeployedsOfTypesContributor;
import com.xebialabs.deployit.plugin.was.contributor.webserver.task.CellTasks;
import com.xebialabs.deployit.plugin.was.deployed.ExtensibleDeployedArtifact;

import static com.google.common.base.Predicates.or;
import static com.google.common.collect.Iterables.concat;
import static com.google.common.collect.Iterables.filter;
import static com.xebialabs.deployit.plugin.was.util.Predicates.instanceOf;
import static java.lang.Boolean.TRUE;

public class GlobalWebServerPluginConfigurationContributor extends CollectDeployedsOfTypesContributor<ExtensibleDeployedArtifact> {
    private static final String CELL_UPDATE_GLOBAL_PLUGIN_PROPERTY = "updateGlobalPlugin";
    private static final Predicate<ConfigurationItem> IS_EAR_OR_WAR_MODULE =
        or(instanceOf(Type.valueOf("was.EarModule")),
            instanceOf(Type.valueOf("was.WarModule")));

    public GlobalWebServerPluginConfigurationContributor() {
        super(Type.valueOf(ExtensibleDeployedArtifact.class));
    }

    /**
     * GenerateAndPropagateGlobalWebServerPlugin#generateAndPropagateWebserverPlugin contributor will add steps to
     * update global plugin configuration for all of the cells (deployment manager or unmanaged servers) with property
     * 'updateGlobalPlugin' set to TRUE.
     *
     * It will:
     *
     * <pre>
     * - update global plugin configuration for all of the cells (deployment manager and unmanaged servers) with property
     * 'updateGlobalPlugin' set to TRUE.
     * - propagate global plugin configuration to all of the unmanaged web servers referenced by the cell (deployment manager or
     * unmanaged server)
     * </pre>
     */
    @Contributor
    public static void generateAndPropagateWebserverPlugin(Deltas deltas, DeploymentPlanningContext ctx) {
        GlobalWebServerPluginConfigurationContributor contributor = new GlobalWebServerPluginConfigurationContributor();
        contributor.filterDeltas(deltas.getDeltas());

        for (Cell cell : contributor.getReferencedCells()) {
            // may be null
            if (TRUE.equals(cell.getProperty(CELL_UPDATE_GLOBAL_PLUGIN_PROPERTY))) {
                CellTasks.updateGlobalPlugin(cell, ctx);
                CellTasks.propagateGlobalPlugin(cell, ctx);
            }
        }
    }

    protected Set<Cell> getReferencedCells() {
        Builder<Cell> cells = ImmutableSet.builder();

        // gather all referenced cells
        for (ExtensibleDeployedArtifact earOrWarModule : getEarAndWarModules()) {
            cells.add(earOrWarModule.getContainer().getCell());
        }
        return cells.build();
    }

    private Set<ExtensibleDeployedArtifact> getEarAndWarModules() {
        Builder<ExtensibleDeployedArtifact> earAndWarModules = ImmutableSet.builder();
        earAndWarModules.addAll(filter(concat(deployedsCreated, deployedsRemoved), IS_EAR_OR_WAR_MODULE));
        Iterable<TypedDelta> earAndWarModuleDeltas = filter(concat(deployedsModified, deployedsNoop),
            new Predicate<TypedDelta>() {
                @Override
                public boolean apply(TypedDelta input) {
                    return IS_EAR_OR_WAR_MODULE.apply(input.getDeployed());
                }
            });
        for (TypedDelta earOrWarModuleDelta : earAndWarModuleDeltas) {
            earAndWarModules.add(earOrWarModuleDelta.getPrevious());
            earAndWarModules.add(earOrWarModuleDelta.getDeployed());
        }
        return earAndWarModules.build();
    }

}
