package com.xebialabs.deployit.plugin.was.deployed;

import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.udm.DeployableArtifact;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.DeployedSpecific;
import com.xebialabs.deployit.plugin.api.udm.artifact.DerivedArtifact;
import com.xebialabs.deployit.plugin.api.validation.Placeholders;
import com.xebialabs.deployit.plugin.was.container.WasContainer;
import com.xebialabs.overthere.OverthereFile;

import java.util.Map;

import static com.google.common.collect.Maps.newHashMap;
import static com.xebialabs.deployit.plugin.was.deployed.ExtensibleDeployedHelper.syncNodes;

@SuppressWarnings("serial")
@Metadata(virtual = true, description = "Base class for all extensible deployed artifact configuration items.")
@Placeholders
public class ExtensibleDeployedArtifact extends ExtensibleDeployed<DeployableArtifact, WasContainer> implements DerivedArtifact<DeployableArtifact> {

    public static final String APP_EDITION_PROPERTY = "appEdition";
    public static final String IS_EDITION_PROPERTY = "isEdition";

    private OverthereFile derivedFile;

    @Property(hidden = true, defaultValue = "Rollout", description = "The word that is used to prefix a step description for the rollout operation.")
    private String rolloutEditionedVerb;

    @Property(hidden = true, defaultValue = "20", description = "The order in which a rollout step will be executed.")
    private int rolloutEditionedOrder;

    @Property(hidden = true, required = false, description = "Python script invoked to rollout a Java EE artifact.")
    private String rolloutEditionedScript;

    @Property(hidden = true, defaultValue = "10", description = "The order in which a create step for Editioned application will be executed(was85nd only).")
    private int createEditionedOrder;

    @Property(hidden = true, defaultValue = "11", description = "The order in which a synchronize after create step for Editioned application will be executed.")
    private int syncAfterCreateEditionedOrder;

    @Property(description = "A Map containing all the placeholders mapped to their values. Special values are <ignore> or <empty>", required = false)
    @DeployedSpecific
    private Map<String, String> placeholders = newHashMap();

    @Override
    public OverthereFile getFile() {
        return derivedFile;
    }

    @Override
    public void setFile(OverthereFile file) {
        this.derivedFile = file;
    }

    @Override
    public DeployableArtifact getSourceArtifact() {
        return getDeployable();
    }

    @Override
    public Map<String, String> getPlaceholders() {
        return placeholders;
    }

    @Override
    public void setPlaceholders(Map<String, String> placeholders) {
        this.placeholders = placeholders;
    }

    protected String getDescription(String verb) {
        if (isEdition(this)) {
            return String.format("%s %s-edition%s on %s", verb, getName(), getProperty(APP_EDITION_PROPERTY), getContainer().getName());
        } else {
            return super.getDescription(verb);
        }
    }

    private boolean isEdition(ExtensibleDeployedArtifact deployedArtifact) {
        return deployedArtifact.hasProperty(IS_EDITION_PROPERTY) && deployedArtifact.<Boolean>getProperty(IS_EDITION_PROPERTY);
    }

    public void createEditioned(DeploymentPlanningContext ctx, Delta delta, CheckpointInfo checkpoint) {
        addStep(ctx, createEditionedOrder, getCreateScript(), getCreateVerb(), checkpoint);
        syncNodes(ctx, syncAfterCreateEditionedOrder, getContainer());
        addStep(ctx, rolloutEditionedOrder, rolloutEditionedScript, rolloutEditionedVerb, false, checkpoint);
    }

    public void modify(DeploymentPlanningContext ctx, Delta delta, CheckpointInfo checkpoint) {
        ExtensibleDeployedArtifact current = (ExtensibleDeployedArtifact) delta.getDeployed();
        String modifyScript = getModifyScript();
        if((current.getType().toString()).equals("was.EarModule") || (current.getType().toString()).equals("was.WarModule")){
            if(current.getProperty("updateStrategy")){
                modifyScript = "was/application/update-application.py";
            }
        }
        if (addStep(ctx, getModifyOrder(), modifyScript, getModifyVerb(), checkpoint)) {
            addStartStep(ctx);
            addStopStep(ctx);
            syncNodes(ctx, getSyncAfterModifyOrder(), getContainer());
        } else {
            ExtensibleDeployedArtifact previous = (ExtensibleDeployedArtifact) delta.getPrevious();

            // Apply the most current start/stop strategy the user has selected.
            previous.setRestartStrategy(current.getRestartStrategy());
            previous.setContainerRestartStrategy(current.getContainerRestartStrategy());

            if (isEdition(current)) {
                current.createEditioned(ctx, delta, checkpoint);
                if (!current.getProperty(APP_EDITION_PROPERTY).equals(previous.getProperty(APP_EDITION_PROPERTY))) {
                    previous.destroy(ctx, delta, checkpoint, false);
                }
            } else {
                if (checkpoint == null) {
                    previous.destroy(ctx, delta, null, true);
                    current.create(ctx, delta, null);
                } else {
                    previous.destroy(ctx, delta);
                    current.create(ctx, delta);
                }
            }
        }
    }
}
