package com.xebialabs.deployit.plugin.was.controltask.dmgr;

import com.google.common.base.Joiner;
import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Delegate;
import com.xebialabs.deployit.plugin.generic.step.SuccessStep;
import com.xebialabs.deployit.plugin.was.container.NodeAgent;

import java.util.List;
import java.util.Map;

public class NodeAgentDelegate {
    private static final String KEY_NODE = "node";
    private static final String KEY_DEPLOYMENT_MANAGER = "deploymentManager";
    private static final String PROPERTY_WASHOME = "wasHome";
    private static final String PROPERTY_HOST = "host";

    @Delegate(name="nodeAgentStart")
    public static List<Step> startNodeAgent(ConfigurationItem ci, String method, Map<String, String> arguments) {
        NodeAgent nodeAgent = populateNodeAgentValues(ci);
        return Lists.newArrayList(createStep(Operation.START, nodeAgent));
    }

    @Delegate(name="nodeAgentStop")
    public static List<Step> stopNodeAgent(ConfigurationItem ci, String method, Map<String, String> arguments) {
        NodeAgent nodeAgent = populateNodeAgentValues(ci);
        return Lists.newArrayList(createStep(Operation.STOP, nodeAgent));
    }

    static Step createStep(Operation operation, NodeAgent nodeAgent) {
        Step step = null;
        List<String> messages = validate(nodeAgent);
        boolean isValid = messages.isEmpty();
        if (isValid) {
            step = StartStopDelegate.createStep(operation, nodeAgent, getContext(nodeAgent));
        } else {
            String details = Joiner.on("").join(messages);
            String desc = String.format("Operation '%s' will not be performed on node agent '%s'. %s", operation, nodeAgent, details);
            step = new SuccessStep(1, desc);
        }

        return step;
    }

    private static List<String> validate(NodeAgent ci) {
        List<String> messages = Lists.newArrayList();
        validateString(messages, ci, PROPERTY_WASHOME);
        validateRef(messages, ci, PROPERTY_HOST);
        return messages;
    }

    private static void validateString(List<String> messages, NodeAgent ci, String property) {
        if (!ci.hasProperty(property) || Strings.isNullOrEmpty(ci.<String>getProperty(property))) {
            String msg = String.format("Property %s is required.", property);
            messages.add(msg);
        }
    }

    private static void validateRef(List<String> messages, NodeAgent ci, String property) {
        Object propValue = ci.getProperty(property);
        if (!ci.hasProperty(property) || null == propValue) {
            String msg = String.format("Property %s is required.", property);
            messages.add(msg);
        }
    }

    private static Map<String, Object> getContext(NodeAgent nodeAgent) {
        Map<String, Object> ctx = Maps.newHashMap();
        ctx.put(KEY_NODE, nodeAgent);
        ctx.put(KEY_DEPLOYMENT_MANAGER, NodeAgent.class.cast(nodeAgent).getCell());
        return ctx;
    }

    /**
     * Populate {@link NodeAgent} with values received as per {@link ConfigurationItem}
     * @param ci - discovered configuration item
     * @return - {@link NodeAgent} object with values set
     */
    private static NodeAgent populateNodeAgentValues(ConfigurationItem ci){
        NodeAgent nodeAgent = (NodeAgent) ci;
        if(nodeAgent.getWasHome() == null){
            nodeAgent.setWasHome(((NodeAgent) ci).getCell().getWasHome());
            nodeAgent.setHost(((NodeAgent) ci).getCell().getHost());
        }
        else {
            nodeAgent.setWasHome(((NodeAgent) ci).getWasHome());
            nodeAgent.setHost(((NodeAgent) ci).getHost());
        }
        return nodeAgent;
    }
}
