package com.xebialabs.deployit.plugin.was.deployed;

import com.xebialabs.deployit.plugin.api.deployment.planning.*;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.udm.Deployable;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.python.PythonManagedDeployed;
import com.xebialabs.deployit.plugin.was.container.WasContainer;
import com.xebialabs.deployit.plugin.was.util.ContainerRestartStrategy;
import com.xebialabs.deployit.plugin.was.util.DeployedRestartStrategy;

import static com.xebialabs.deployit.plugin.api.deployment.specification.Operation.*;
import static com.xebialabs.deployit.plugin.was.deployed.ExtensibleDeployedHelper.syncNodes;

@SuppressWarnings("serial")
@Metadata(description = "Base class for all extensible deployed configuration items.")
public abstract class ExtensibleDeployed<D extends Deployable, C extends WasContainer> extends PythonManagedDeployed<D, C> {

    @Property(hidden = false, required = false, description = "The name used by Websphere")
    private String wasName;

    @Property(hidden = true, required = false, description = "Python script invoked to deploy a Java EE artifact or create a Java EE resource")
    private String createScript;

    @Property(hidden = true, defaultValue = "Create", description = "The word that is used to prefix a step description for the create operation.")
    private String createVerb;

    @Property(hidden = true, defaultValue = "60", description = "The order in which a create step will be executed.")
    private int createOrder;

    @Property(hidden = true, defaultValue = "65", description = "The order in which a synchronize after create step will be executed.")
    private int syncAfterCreateOrder;

    @Property(hidden = true, required = false, description = "Python script invoked to upgrade a Java EE artifact or modify a Java EE resource")
    private String modifyScript;

    @Property(hidden = true, defaultValue = "Modify", description = "The word that is used to prefix a step description for the modify operation.")
    private String modifyVerb;

    @Property(hidden = true, defaultValue = "60", description = "The order in which a modify step will be executed.")
    private int modifyOrder;

    @Property(hidden = true, defaultValue = "65", description = "The order in which a synchronize after modify step will be executed.")
    private int syncAfterModifyOrder;

    @Property(hidden = true, required = false, description = "Python script invoked to undeploy a Java EE artifact or destroy a Java EE resource")
    private String destroyScript;

    @Property(hidden = true, defaultValue = "Destroy", description = "The word that is used to prefix a step description for the destroy operation.")
    private String destroyVerb;

    @Property(hidden = true, defaultValue = "40", description = "The order in which a destroy step will be executed.")
    private int destroyOrder;

    @Property(required = false, hidden = true, description = "Python script invoked for the noop operation.")
    private String noopScript;

    @Property(hidden = true, defaultValue = "Noop")
    private String noopVerb;

    @Property(hidden = true, defaultValue = "50", description = "The order of the step in the step list for the noop operation.")
    private int noopOrder;

    @Property(hidden = true, defaultValue = "45", description = "The order in which a synchronize after destroy step will be executed.")
    private int syncAfterDestroyOrder;

    @Property(hidden = true, required = false, description = "Python script invoked to start a Java EE artifact or Java EE resource")
    private String startScript;

    @Property(hidden = true, defaultValue = "Start", description = "The word that is used to prefix a step description for the start operation.")
    private String startVerb;

    @Property(hidden = true, defaultValue = "50", description = "The order in which a start step will be executed.")
    private int startOrder;

    @Property(hidden = true, required = false, description = "Python script invoked to stop a Java EE artifact or Java EE resource")
    private String stopScript;

    @Property(hidden = true, defaultValue = "Stop", description = "The word that is used to prefix a step description for the stop operation.")
    private String stopVerb;

    @Property(hidden = true, defaultValue = "50", description = "The order in which a stop step will be executed.")
    private int stopOrder;

    @Property(defaultValue = "STOP_START", hidden=true, category = "Restart Strategy", description = "The restart strategy for the deployed. " +
            "STOP_START : stop, undeploy, deploy, start. STOP: stop, undeploy, deploy. START: undeploy, deploy, start.")
    private DeployedRestartStrategy restartStrategy = DeployedRestartStrategy.STOP_START;

    @Property(defaultValue = "NONE", hidden=true, category = "Restart Strategy", description = "The restart strategy for the container affected by the deployed. " +
            "STOP_START : stop container, undeploy, deploy, start container. RESTART : undeploy, deploy, restart container.")
    private ContainerRestartStrategy containerRestartStrategy = ContainerRestartStrategy.NONE;

    @Property(defaultValue = "false", required = false, hidden = true, category = "Restart Strategy", description = "When true, the restart strategy is apply for a NOOP operation")
    private boolean applyRestartStrategyOnNoop;

    @Property(defaultValue = "false", required = false, hidden = true, category = "Restart Strategy", description = "If set, each server in the cluster will be first stopped and restarted sequentially.")
    private boolean enableRippleStart;

    @Property(hidden = true, defaultValue = "id, type, properties, deployable, container, createOrder, createScript, createVerb, " +
            "syncAfterCreateOrder, modifyOrder, modifyScript, modifyVerb, noopOrder, noopScript, noopVerb, syncAfterModifyOrder, destroyOrder, destroyScript, " +
            "securityPermissions, inheritPermissions, exposeDeployedApplication, destroyVerb, syncAfterDestroyOrder, startOrder, startScript, startVerb, stopOrder, " +
            "stopScript, stopVerb, discoverScript, inspectScript, discoverOrder, libraryScripts, retryAttemptsForAppReadyCheck, restartStrategy, " +
            "containerRestartStrategy,applyRestartStrategyOnNoop, enableRippleStart, provisioners, ordinal, boundConfigurationItems, wasName",
            description = "Standard properties that are not exposed to any python wsadmin script.")
    private String standardPropertiesNotToExpose;

    @Create
    public void create(DeploymentPlanningContext ctx, Delta delta) {
        create(ctx, delta, checkpoint(delta, CREATE));
    }

    public void create(DeploymentPlanningContext ctx, Delta delta, CheckpointInfo checkpoint) {
        addStep(ctx, createOrder, createScript, createVerb, checkpoint);
        syncNodes(ctx, syncAfterCreateOrder, getContainer());
        addStartStep(ctx);
    }

    protected void addStartStep(DeploymentPlanningContext ctx) {
        if (restartStrategy == DeployedRestartStrategy.STOP_START || restartStrategy == DeployedRestartStrategy.START) {
            addStep(ctx, startOrder, startScript, startVerb, false);
        }
    }

    protected void addStopStep(DeploymentPlanningContext ctx) {
        if (restartStrategy == DeployedRestartStrategy.STOP_START || restartStrategy == DeployedRestartStrategy.STOP) {
            addStep(ctx, stopOrder, stopScript, stopVerb, false);
        }
    }

    @Modify
    public void modify(DeploymentPlanningContext ctx, Delta delta) {
        modify(ctx, delta, checkpoint(delta, MODIFY));
    }

    public void modify(DeploymentPlanningContext ctx, Delta delta, CheckpointInfo checkpoint) {
        if (addStep(ctx, modifyOrder, modifyScript, modifyVerb, checkpoint)) {
            addStartStep(ctx);
            addStopStep(ctx);
            syncNodes(ctx, syncAfterModifyOrder, getContainer());
        } else {
            ExtensibleDeployed<?, ?> previous = (ExtensibleDeployed<?, ?>) delta.getPrevious();
            ExtensibleDeployed<?, ?> current = (ExtensibleDeployed<?, ?>) delta.getDeployed();
            // Apply the most current start/stop strategy the user has selected.
            previous.restartStrategy = current.restartStrategy;
            previous.containerRestartStrategy = current.containerRestartStrategy;
            /*
             * If this method is called with a 'null' checkpoint, make sure that the destroy/create sequence also does
             * not use a checkpoint.
             */
            if (checkpoint == null) {
                previous.destroy(ctx, delta, null, true);
                current.create(ctx, delta, null);
            } else {
                previous.destroy(ctx, delta);
                current.create(ctx, delta);
            }
        }
    }

    @Noop
    @SuppressWarnings("unchecked")
    public void executeNoop(DeploymentPlanningContext ctx, Delta delta) {
        ExtensibleDeployed<D, C> deployed = (ExtensibleDeployed<D, C>) delta.getDeployed();
        deployed.addStep(ctx, noopOrder, noopScript, noopVerb, false, null);
        if (applyRestartStrategyOnNoop) {
            addStartStep(ctx);
            addStopStep(ctx);
        }
    }

    @Destroy
    public void destroy(DeploymentPlanningContext ctx, Delta delta) {
        destroy(ctx, delta, checkpoint(delta, DESTROY), true);
    }

    public void destroy(DeploymentPlanningContext ctx, Delta delta, CheckpointInfo checkpoint, boolean isStopStepRequired) {
        if (isStopStepRequired) {
            addStopStep(ctx);
        }
        addStep(ctx, destroyOrder, destroyScript, destroyVerb, false, checkpoint);
        syncNodes(ctx, syncAfterDestroyOrder, getContainer());
    }

    public String getCreateScript() {
        return createScript;
    }

    public String getCreateVerb() {
        return createVerb;
    }

    public int getCreateOrder() {
        return createOrder;
    }

    public String getModifyScript() {
        return modifyScript;
    }

    public String getModifyVerb() {
        return modifyVerb;
    }

    public int getModifyOrder() {
        return modifyOrder;
    }

    public int getSyncAfterModifyOrder() {
        return syncAfterModifyOrder;
    }

    public String getDestroyScript() {
        return destroyScript;
    }

    public String getDestroyVerb() {
        return destroyVerb;
    }

    public int getDestroyOrder() {
        return destroyOrder;
    }

    public int getSyncAfterDestroyOrder() {
        return syncAfterDestroyOrder;
    }

    public String getStartScript() {
        return startScript;
    }

    public String getStartVerb() {
        return startVerb;
    }

    public int getStartOrder() {
        return startOrder;
    }

    public String getStopScript() {
        return stopScript;
    }

    public String getStopVerb() {
        return stopVerb;
    }

    public int getStopOrder() {
        return stopOrder;
    }

    public String getStandardPropertiesNotToExpose() {
        return standardPropertiesNotToExpose;
    }

    public int getSyncAfterCreateOrder() {
        return syncAfterCreateOrder;
    }

    protected DeployedRestartStrategy getRestartStrategy() {
        return restartStrategy;
    }

    protected void setRestartStrategy(DeployedRestartStrategy restartStrategy) {
        this.restartStrategy = restartStrategy;
    }

    protected ContainerRestartStrategy getContainerRestartStrategy() {
        return containerRestartStrategy;
    }

    protected void setContainerRestartStrategy(ContainerRestartStrategy containerRestartStrategy) {
        this.containerRestartStrategy = containerRestartStrategy;
    }

    @Override
    public String getName() {
        if(wasName != null && !wasName.equals("")){
            return wasName;
        } else {
            return super.getName();
        }
    }

    public String getWasName() {
        return wasName;
    }

    public void setWasName(String wasName) {
        this.wasName = wasName;
    }
}
