package com.xebialabs.deployit.plugin.was.contributor;

import java.util.Set;

import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;

import com.xebialabs.deployit.plugin.api.deployment.planning.Contributor;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.Deltas;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.python.PythonDeploymentStep;
import com.xebialabs.deployit.plugin.was.container.WasContainer;
import com.xebialabs.deployit.plugin.was.util.ContainerRestartStrategy;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static com.google.common.collect.Sets.newHashSet;
import static java.lang.String.format;

public class StopStartContainer {

    protected static final String STOP_SCRIPT_PROPERTY = "stopScript";
    protected static final String START_SCRIPT_PROPERTY = "startScript";
    protected static final String RESTART_SCRIPT_PROPERTY = "restartScript";
    protected static final String STOP_ORDER_PROPERTY = "stopOrder";
    protected static final String START_ORDER_PROPERTY = "startOrder";
    protected static final String RESTART_ORDER_PROPERTY = "restartOrder";
    public static final String CONTAINER_RESTART_STRATEGY = "containerRestartStrategy";
    public static final String CONTAINER_NOOP_RESTART_STRATEGY = "applyRestartStrategyOnNoop";
    public static final String ENABLE_RIPPLE_START = "enableRippleStart";
    public static final String UNMANAGED_SERVER = "UnmanagedServer";
    private static final Logger log = LoggerFactory.getLogger(StopStartContainer.class);

    /**
     * StopStartContainer#stopAndStartContainers contributor will add steps to restart(or stop/start) deployed container
     * if deployed has property 'containerRestartStrategy' with value set to RESTART or STOP_START.
     */
    @Contributor
    public static void stopAndStartContainers(Deltas deltas, DeploymentPlanningContext ctx) {
        StopStartContainer contributor = new StopStartContainer();
        contributor.doStopAndStartContainers(deltas, ctx);
    }

    protected void doStopAndStartContainers(Deltas deltas, DeploymentPlanningContext ctx) {
        Set<WasContainer> stopStartContainers = newHashSet();
        Set<WasContainer> restartStartContainers = newHashSet();

        findContainers(deltas, stopStartContainers, restartStartContainers);
        for (WasContainer wasContainer : stopStartContainers) {
            if (!isRippleStartEnabled(wasContainer)) {
                addStep(STOP_SCRIPT_PROPERTY, STOP_ORDER_PROPERTY, "Stop", wasContainer, ctx);
                addStep(START_SCRIPT_PROPERTY, START_ORDER_PROPERTY, "Start", wasContainer, ctx);
            } else {
                addStep(START_SCRIPT_PROPERTY, START_ORDER_PROPERTY, "RippleStart", wasContainer, ctx);
            }
        }

        for (WasContainer wasContainer : restartStartContainers) {
            if (hasRestartScript(wasContainer)) {
                addStep(RESTART_SCRIPT_PROPERTY, RESTART_ORDER_PROPERTY, "Restart", wasContainer, ctx);
            } else {
                if (!isRippleStartEnabled(wasContainer)) {
                    addStep(STOP_SCRIPT_PROPERTY, RESTART_ORDER_PROPERTY, "Stop", wasContainer, ctx);
                    addStep(START_SCRIPT_PROPERTY, RESTART_ORDER_PROPERTY, "Start", wasContainer, ctx);
                } else {
                    log.warn("Ripple start enabled.");
                    addStep(START_SCRIPT_PROPERTY, RESTART_ORDER_PROPERTY, "RippleStart", wasContainer, ctx);
                }
            }
        }
    }

    protected void addStep(String scriptProperty, String orderProperty, String verb, WasContainer container, DeploymentPlanningContext ctx) {
        if (container.hasProperty(scriptProperty) && container.hasProperty(orderProperty)) {
            ctx.addStep(new PythonDeploymentStep(
                    container.<Integer>getProperty(orderProperty),
                    container.getCell(),
                    container.<String>getProperty(scriptProperty),
                    ImmutableMap.<String, Object>of("container", container),
                    format("%s '%s'", verb, container.getName())));
        }
    }

    protected boolean hasRestartScript(WasContainer container) {
        return container.hasProperty(RESTART_SCRIPT_PROPERTY) && !Strings.isNullOrEmpty(container.<String>getProperty(RESTART_SCRIPT_PROPERTY));
    }

    protected void findContainers(Deltas deltas, Set<WasContainer> stopStartContainers, Set<WasContainer> restartStartContainers) {
        for (Delta delta : deltas.getDeltas()) {
            Deployed<?, ?> deployed = delta.getOperation() == Operation.DESTROY ? delta.getPrevious() : delta.getDeployed();
            if (deployed.getContainer() instanceof WasContainer) {
                WasContainer container = (WasContainer) deployed.getContainer();
                if (isRippleStartEnabled(deployed)) {
                    container.setProperty(ENABLE_RIPPLE_START, true);
                }
                if (!addContainerDependingOnStrategy(delta, deployed, container, stopStartContainers, restartStartContainers)) {
                    addContainerDependingOnStrategy(delta, container, stopStartContainers, restartStartContainers);
                }
            }
        }
        restartStartContainers.removeAll(stopStartContainers); // stop_start takes precedence over restart
    }

    protected boolean addContainerDependingOnStrategy(Delta delta, WasContainer container, Set<WasContainer> stopStartContainers, Set<WasContainer> restartStartContainers) {
        return addContainerDependingOnStrategy(delta, container, container, stopStartContainers, restartStartContainers);
    }

    protected boolean addContainerDependingOnStrategy(Delta delta, ConfigurationItem ci, WasContainer container, Set<WasContainer> stopStartContainers, Set<WasContainer> restartStartContainers) {
        if (isOperationNoop(delta.getOperation()) && !applyRestartStrategyForNoop(ci)) {
            return false;
        }
        if (hasContainerRestartStrategy(ci)) {
            if (isStopStartStrategy(ci)) {
                stopStartContainers.add(container);
                return true;
            } else if (isRestartStrategy(ci)) {
                restartStartContainers.add(container);
                return true;
            }
        }
        return false;
    }

    protected boolean applyRestartStrategyForNoop(ConfigurationItem ci) {
        return hasContainerNoopRestartStrategy(ci) && isNoopRestartStrategy(ci);
    }

    protected boolean hasContainerRestartStrategy(ConfigurationItem configurationItem) {
        return configurationItem.hasProperty(CONTAINER_RESTART_STRATEGY) && configurationItem.getProperty(CONTAINER_RESTART_STRATEGY) != ContainerRestartStrategy.NONE;
    }

    private boolean hasContainerNoopRestartStrategy(ConfigurationItem configurationItem) {
        return configurationItem.hasProperty(CONTAINER_NOOP_RESTART_STRATEGY);
    }

    protected boolean isRestartStrategy(ConfigurationItem ci) {
        return ci.getProperty(CONTAINER_RESTART_STRATEGY) == ContainerRestartStrategy.RESTART;
    }

    protected boolean isStopStartStrategy(ConfigurationItem ci) {
        return ci.getProperty(CONTAINER_RESTART_STRATEGY) == ContainerRestartStrategy.STOP_START;
    }

    private boolean isNoopRestartStrategy(ConfigurationItem ci) {
        return ci.getProperty(CONTAINER_NOOP_RESTART_STRATEGY);
    }

    private boolean isRippleStartEnabled(ConfigurationItem ci) {
        if (ci.getType().getName().equals(UNMANAGED_SERVER)) {
            return false;
        }

        return ci.getProperty(ENABLE_RIPPLE_START);
    }

    protected boolean isOperationNoop(Operation deltaOperation) {
        return deltaOperation == Operation.NOOP;
    }

}
