package com.xebialabs.deployit.plugin.cloud.vsphere.steps;

import java.rmi.RemoteException;
import java.util.List;
import java.util.concurrent.Callable;
import com.vmware.vim25.TaskInfo;
import com.vmware.vim25.TaskInfoState;
import com.vmware.vim25.mo.Task;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.cloud.step.ContextAttribute;
import com.xebialabs.deployit.plugin.cloud.util.Retrier;
import com.xebialabs.deployit.plugin.cloud.vsphere.access.VsphereAdapter;
import com.xebialabs.deployit.plugin.cloud.vsphere.ci.HostTemplate;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.vmware.vim25.TaskInfoState.error;

@SuppressWarnings("unchecked")
public class WaitForVsphereTasksStep implements Step {

    private VsphereAdapter adapter;

    private HostTemplate template;

    public WaitForVsphereTasksStep(HostTemplate template, VsphereAdapter adapter) {
        this.adapter = adapter;
        this.template = template;
    }

    @Override
    public int getOrder() {
        return DEFAULT_ORDER;
    }

    @Override
    public String getDescription() {
        return "Wait for vSphere tasks to complete";
    }

    @Override
    public StepExitCode execute(final ExecutionContext ctx) throws Exception {

        final List<String> startedTasks = checkNotNull((List<String>)ctx.getAttribute(ContextAttribute.STARTED_TASKS.name()));

        ctx.logOutput(startedTasks.size() + " vSphere task(s) were started earlier.");

        if (startedTasks.isEmpty()) {
            return StepExitCode.SUCCESS;
        }

        for (String t : startedTasks) {
            final Task task = adapter.getTask(t);

            Boolean ok = new Retrier<Boolean>(new Callable<Boolean>() {
                @Override
                public Boolean call() throws Exception {

                    printTasksStatus(ctx, startedTasks);

                    if (task.getTaskInfo().getState() == TaskInfoState.success) {
                        return true;
                    }

                    if (task.getTaskInfo().isCancelled()) {
                        ctx.logError("The task has been cancelled");
                        return false;
                    }

                    if (task.getTaskInfo().getState() == error) {
                        ctx.logError("There was and error during the task execution : " + task.getTaskInfo().getError().getLocalizedMessage());
                        return false;
                    }

                    throw new IllegalStateException("Task is in state " + task.getTaskInfo().getState() + ". Waiting.");
                }
            }).retryFor(template.getBootTimeout(), template.getRetryDelay());

            if (!ok) {
                return StepExitCode.FAIL;
            }
        }

        ctx.setAttribute(ContextAttribute.STARTED_TASKS.name(), null);

        return StepExitCode.SUCCESS;
    }

    private void printTasksStatus(ExecutionContext ctx, List<String> tasks) throws RemoteException {
        for (String taskId : tasks) {
            TaskInfo taskInfo = adapter.getTask(taskId).getTaskInfo();
            TaskInfoState state = taskInfo.getState();

            int p;

            if (state == TaskInfoState.success) {
                p = 100;
            } else if (taskInfo.getProgress() == null) {
                p = 0;
            } else {
                p = taskInfo.getProgress();
            }

            ctx.logOutput("Task progress of " + taskId + ": " + p + "%");
        }
    }
}
