package com.xebialabs.deployit.plugin.trigger.action;

import com.google.common.base.Strings;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.deployit.plugin.trigger.Action;
import com.xebialabs.deployit.plugin.generic.freemarker.ConfigurationHolder;
import com.xebialabs.deployit.plugin.mail.SmtpServer;
import freemarker.template.Configuration;
import freemarker.template.Template;
import freemarker.template.TemplateException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.mail.MessagingException;
import java.io.*;
import java.util.List;
import java.util.Map;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.plugin.generic.freemarker.ConfigurationHolder.resolveExpression;

@Metadata(root = Metadata.ConfigurationItemRoot.CONFIGURATION, description = "Email Action")
public class EmailNotification extends BaseConfigurationItem implements Action {

    @Property(description = "Mail addresses of recepients.")
    private List<String> toAddresses = newArrayList();

    @Property(required = false, description = "From mail address. Defaults to SMTPServer fromAddress.")
    private String fromAddress;

    @Property(description = "Mail subject")
    private String subject;

    @Property(required = false, size = Property.Size.LARGE, description = "Mail body content in the form of a Freemarker template.")
    private String body;

    @Property(required =false, description = "Freemarker template used to render mail body content. Path can be absolute or relative to Deployit's classpath.")
    private String bodyTemplatePath;

    @Property(description = "The mail server used to send the email.")
    private SmtpServer mailServer;

    public SmtpServer getMailServer() {
        return mailServer;
    }

    public void setMailServer(SmtpServer mailServer) {
        this.mailServer = mailServer;
    }

    public String getBodyTemplatePath() {
        return bodyTemplatePath;
    }

    public void setBodyTemplatePath(String bodyTemplatePath) {
        this.bodyTemplatePath = bodyTemplatePath;
    }

    @Override
    public void execute(Map<String,Object> ctx) {
        try {
            String msg;
            if (Strings.isNullOrEmpty(body)) {
                msg = renderEmailTemplate(bodyTemplatePath, ctx, false);
            } else {
                msg = renderEmailTemplate(body, ctx, true);
            }

            if (!Strings.isNullOrEmpty(msg.trim())) {
                String resolvedSubject = resolveExpression(subject,ctx);
                String resolvedFromAddress= resolveExpression(fromAddress,ctx);
                List<String> resolvedRecipients = resolveExpression(toAddresses, ctx);
                mailServer.sendMessage(resolvedSubject,msg,resolvedRecipients,resolvedFromAddress);
            }
        } catch (TemplateException e) {
            logger.error("Failed to render email template.", e);
        } catch (IOException e) {
            logger.error("Failed to load email template.", e);
        } catch (MessagingException e) {
            logger.error("Failed to send mail.", e);
        }
    }

    protected String renderEmailTemplate(String template, Map<String, Object> vars, boolean inlineTemplate) throws IOException, TemplateException {
        Configuration cfg = ConfigurationHolder.getConfiguration();
        Template loadedTemplate;
        if (inlineTemplate) {
            loadedTemplate = new Template("name", new StringReader(template),cfg);
        } else {
            //check if absolute path
            File templateFile = new File(template);
            if (templateFile.exists()) {
                loadedTemplate = new Template("name", new FileReader(template),cfg);
            } else {
                //resolve via classpath.
                loadedTemplate = cfg.getTemplate(template);
            }
        }
        StringWriter sw = new StringWriter();
        loadedTemplate.process(vars, sw);
        return sw.toString();
    }

    public List<String> getToAddresses() {
        return toAddresses;
    }

    public void setToAddresses(List<String> toAddresses) {
        this.toAddresses = toAddresses;
    }

    public String getFromAddress() {
        return fromAddress;
    }

    public void setFromAddress(String fromAddress) {
        this.fromAddress = fromAddress;
    }

    public String getSubject() {
        return subject;
    }

    public void setSubject(String subject) {
        this.subject = subject;
    }

    public String getBody() {
        return body;
    }

    public void setBody(String body) {
        this.body = body;
    }

    private static final Logger logger = LoggerFactory.getLogger(EmailNotification.class);
}
