package com.xebialabs.deployit.plugin.tomcat.processor;

import java.util.Collections;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;

import com.xebialabs.deployit.plugin.api.deployment.planning.PrePlanProcessor;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.generic.deployed.AbstractDeployed;
import com.xebialabs.deployit.plugin.tomcat.deployed.ContextElement;
import com.xebialabs.deployit.plugin.tomcat.deployed.ContextWarModule;

import static com.google.common.base.Preconditions.checkState;
import static com.google.common.base.Strings.emptyToNull;
import static com.google.common.collect.Iterables.filter;
import static com.xebialabs.deployit.plugin.tomcat.processor.Utils.extractDeployedContextElements;
import static com.xebialabs.deployit.plugin.tomcat.processor.Utils.findDeployedsTargetedToContainer;
import static com.xebialabs.deployit.plugin.tomcat.processor.Utils.isSubtypeOf;

public class ApplicationContextNameInjector {

    public static final Type APPLICATION_CONTEXT_TYPE = Type.valueOf("tomcat.VirtualHost");

    @PrePlanProcessor
    public List<Step> injectContextNameInToContextElements(DeltaSpecification spec) {
        Iterable<ContextElement<?>> contextElements = findAllContextElementsTargetedToApplicationContexts(spec.getDeltas());
        logger.debug("Found {} context elements targeted to application contexts", Iterables.size(contextElements));
        Iterable<Delta> warModules = findContextWarModulesTargetedToVirtualHost(spec.getDeltas());
        logger.debug("Found {} warModules targeted to application contexts", Iterables.size(warModules));
        injectContextNameIntoContextElementsUsingWarResolution(warModules, contextElements);
        validateAllContextElementsTargetedToApplicationContextHasCtxNameSet(contextElements);
        return Collections.emptyList();
    }

    protected void injectContextNameIntoContextElementsUsingWarResolution(Iterable<Delta> warModules, Iterable<ContextElement<?>> contextElements) {
        if (Iterables.size(warModules) == 1) {
            Delta warModuleDelta = warModules.iterator().next();
            injectContextIntoContextElementsUsingWar(warModuleDelta, contextElements);
        }
    }

    protected void validateAllContextElementsTargetedToApplicationContextHasCtxNameSet( Iterable<ContextElement<?>> contextElements) {
        for (ContextElement<?> ce : contextElements) {
            checkState(emptyToNull(ce.getContext()) != null, "Context for resource %s targeted to %s cannot be determined. Please specify the context.", ce.getName(), ce.getContainer().getName());
        }
    }

    protected void injectContextIntoContextElementsUsingWar(Delta warModuleDelta, Iterable<ContextElement<?>> contextElements) {
        ContextWarModule warModule = (ContextWarModule) warModuleDelta.getDeployed();
        logger.debug("Only a single war targeted to application context. All contextElements without a context will be associated with '{}'", warModule.getContextRoot());
        if (warModuleDelta.getOperation() == Operation.CREATE) {
            for (ContextElement<?> contextElement : contextElements) {
                if (emptyToNull(contextElement.getContext()) == null) {
                    contextElement.setContext(warModule.getContextRoot());
                }
            }
        } else if (warModuleDelta.getPrevious() != null) { //defensive, in case subclasses pass in a destroy operation delta.
            ContextWarModule oldWarModule = (ContextWarModule) warModuleDelta.getPrevious();
            for (ContextElement<?> contextElement : contextElements) {
                if (oldWarModule.getContextRoot().equals(contextElement.getContext()) || emptyToNull(contextElement.getContext()) == null) {
                    contextElement.setContext(warModule.getContextRoot());
                }
            }
        }
    }

    protected Iterable<ContextElement<?>> findAllContextElementsTargetedToApplicationContexts(List<Delta> deltas) {
        Iterable<AbstractDeployed<?>> deployeds = findDeployedsTargetedToContainer(deltas, APPLICATION_CONTEXT_TYPE);
        return extractDeployedContextElements(deployeds);
    }

    protected static Iterable<Delta> findContextWarModulesTargetedToVirtualHost(List<Delta> deltas) {
        return filter(deltas, new Predicate<Delta>() {
            @Override
            public boolean apply(Delta input) {
                if (input.getOperation() != Operation.DESTROY) {
                    return accept(input.getDeployed());
                }
                return false;
            }

            private boolean accept(Deployed<?, ?> deployed) {
                return deployed != null && isSubtypeOf(APPLICATION_CONTEXT_TYPE, deployed.getContainer().getType()) &&
                        deployed instanceof ContextWarModule;
            }
        });


    }

    private static final Logger logger = LoggerFactory.getLogger(ApplicationContextNameInjector.class);

}
