package com.xebialabs.deployit.plugin.tomcat.deployed;

import java.util.Set;

import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.generic.deployed.CopiedArtifact;
import com.xebialabs.deployit.plugin.generic.step.ArtifactDeleteStep;
import com.xebialabs.deployit.plugin.generic.step.TemplateArtifactCopyStep;
import com.xebialabs.deployit.plugin.jee.artifact.War;
import com.xebialabs.deployit.plugin.tomcat.step.RenameFileStep;

import static com.google.common.collect.Sets.newHashSet;

@SuppressWarnings("serial")
@Metadata(virtual = true, description = "War Module that has an associated context.xml")
public class ContextWarModule extends CopiedArtifact<War> {

    @Property(description = "Context root under which the web application can be access", defaultValue = "${deployed.name}")
    private String contextRoot;

    @Property(hidden = true, description = "Freemarker template used to create the context xml for the War")
    private String contextTemplate;

    @Property(hidden = true, description = "Target directory to which the context xml must to copied to on the target system")
    private String contextXmlTargetDirectory;

    @Property(hidden = true, required = false, defaultValue = "true", description = "If set to true, create and destroy context.xml files. Set to false to provide your own context.xml file.")
    private boolean manageContextXml;

    @Property(hidden = true, required = false, defaultValue = "false", description = "If set to false, keep context file on destroy. Set to true to destroy the context file.")
    private boolean removeContextXml;

    @Property(hidden = true, defaultValue = "60", description = "The order of the create context step in the step list for the create operation.")
    private int createOrderOfContextXml;

    @Property(hidden = true, defaultValue = "60", description = "The order of the rename context root step in the step list for the modify operation.")
    private int modifyOrderOfContextXml;

    @Property(hidden = true, defaultValue = "40", description = "The order of the destroy context step in the step list for the create operation.")
    private int destroyOrderOfContextXml;

    @Property(hidden = true, defaultValue = "war settings")
    private String contextSubject;

    @Property(required = false, hidden = true, description = "Absolute paths to files on the target system that was generated and must be clean up during a destroy operation")
    private Set<String> deleteGeneratedResources = newHashSet();

    private boolean destroyOperation;

    @Override
    public void executeModify(DeploymentPlanningContext ctx, Delta delta) {
        ContextWarModule old = (ContextWarModule) delta.getPrevious();
        if (manageContextXml && !old.getContextRoot().equals(getContextRoot())) {
            String fileSeparator = getContainer().getHost().getOs().getFileSeparator();
            String source = old.resolveContextFile(fileSeparator);
            String target = resolveContextFile(fileSeparator);
            String description = "Change context from " + old.getContextRoot() + " to " + getContextRoot() + " on " + getContainer().getName();
            ctx.addStep(new RenameFileStep(getModifyOrderOfContextXml(),source, target, description, getContainer()));
        }
        ctx.addStepWithCheckpoint(old.createArtifactDeleteStep(), delta, Operation.DESTROY);
        if(manageContextXml) {
            ctx.addStep(createTemplateArtifactCopyStep(getModifyOrderOfContextXml(), getModifyVerb()));
        }
        ctx.addStepWithCheckpoint(createArtifactCopyStep(), delta, Operation.CREATE);
    }

    @Override
    public void executeCreate(DeploymentPlanningContext ctx, Delta delta) {
        if(manageContextXml) {
            ctx.addStep(createTemplateArtifactCopyStep(getCreateOrderOfContextXml(), getCreateVerb()));
        }
        ctx.addStepWithCheckpoint(createArtifactCopyStep(), delta, Operation.CREATE);
    }

    @Override
    public void executeDestroy(DeploymentPlanningContext ctx, Delta delta) {
        destroyOperation = true;
        ctx.addStep(createArtifactDeleteStep());
        if(manageContextXml) {
            ctx.addStepWithCheckpoint(createTemplateArtifactCopyStep(getDestroyOrderOfContextXml(), getModifyVerb()), delta, Operation.DESTROY);
        }
    }

    protected TemplateArtifactCopyStep createTemplateArtifactCopyStep(int order, String verb) {
        TemplateArtifactCopyStep step = new TemplateArtifactCopyStep(order, getContainer(), getDeployedAsFreeMarkerContext(), getContextTemplate(), getContextXmlTargetDirectory());
        step.setCreateTargetPath(isCreateTargetDirectory());
        step.setTargetFileName(getContextRoot() + ".xml");
        step.setSourceFileDescription("Context Xml");
        step.setDescription(getContextDescription(verb));
        return step;
    }

    @Override
    protected ArtifactDeleteStep createArtifactDeleteStep() {
        ArtifactDeleteStep step = super.createArtifactDeleteStep();
        step.setDeleteAdditionalTargetFiles(newHashSet(getTargetDirectory() + step.getHostFileSeparator() + getContextRoot()));
        return step;
    }

    public String resolveContextFile(String fileSeparator) {
        return getContextXmlTargetDirectory() + fileSeparator + getContextRoot() + ".xml";
    }

    public String getContextDescription(String verb) {
        return String.format("%s %s in %s.xml on %s", verb, getContextSubject(), getContextRoot(), getContainer().getName());
    }

    public String getContextRoot() {
        return resolveExpression(contextRoot);
    }

    public void setContextRoot(String contextRoot) {
        this.contextRoot = contextRoot;
    }

    public String getContextTemplate() {
        return resolveExpression(contextTemplate);
    }

    public void setContextTemplate(String contextTemplate) {
        this.contextTemplate = contextTemplate;
    }

    public int getCreateOrderOfContextXml() {
        return createOrderOfContextXml;
    }

    public void setCreateOrderOfContextXml(int createOrderOfContextXml) {
        this.createOrderOfContextXml = createOrderOfContextXml;
    }

    public String getContextXmlTargetDirectory() {
        return resolveExpression(contextXmlTargetDirectory);
    }

    public void setContextXmlTargetDirectory(String contextXmlTargetDirectory) {
        this.contextXmlTargetDirectory = contextXmlTargetDirectory;
    }

    public boolean isRemoveContextXml() {
        return removeContextXml;
    }

    public void setRemoveContextXml(boolean removeContextXml) {
        this.removeContextXml = removeContextXml;
    }

    public Set<String> getDeleteGeneratedResources() {
        return resolveExpression(deleteGeneratedResources);
    }

    public void setDeleteGeneratedResources(Set<String> deleteGeneratedResources) {
        this.deleteGeneratedResources = deleteGeneratedResources;
    }

    public int getModifyOrderOfContextXml() {
        return modifyOrderOfContextXml;
    }

    public void setModifyOrderOfContextXml(int modifyOrderOfContextXml) {
        this.modifyOrderOfContextXml = modifyOrderOfContextXml;
    }

    public int getDestroyOrderOfContextXml() {
        return destroyOrderOfContextXml;
    }

    public void setDestroyOrderOfContextXml(int destroyOrderOfContextXml) {
        this.destroyOrderOfContextXml = destroyOrderOfContextXml;
    }

    public boolean isDestroyOperation() {
        return destroyOperation;
    }

    public String getContextSubject() {
        return contextSubject;
    }

    public void setContextSubject(String contextSubject) {
        this.contextSubject = contextSubject;
    }

    public boolean isManageContextXml() {
        return manageContextXml;
    }

    public void setManageContextXml(final boolean manageContextXml) {
        this.manageContextXml = manageContextXml;
    }
}
