package com.xebialabs.xldeploy.authentication.oidc.conf

import ai.digital.deploy.sql.model.LoginProvider
import com.nimbusds.jose.jwk._
import com.nimbusds.jose.{Algorithm, JWSAlgorithm}
import com.typesafe.config.{Config, ConfigException}
import com.xebialabs.deployit.ServerConfiguration
import com.xebialabs.deployit.booter.local.utils.Strings.{isBlank, isNotBlank}
import com.xebialabs.deployit.checks.Checks.checkArgument
import com.xebialabs.deployit.engine.spi.exception.DeployitException
import com.xebialabs.platform.sso.oidc.authentication.CustomOidcIdTokenDecoderFactory
import com.xebialabs.platform.sso.oidc.exceptions.UnsupportedOidcConfigurationException
import com.xebialabs.platform.sso.oidc.policy.ClaimsToGrantedAuthoritiesPolicy
import com.xebialabs.platform.sso.oidc.policy.impl.{DefaultClaimsToGrantedAuthoritiesPolicy, GrantedAuthoritiesExtractor}
import com.xebialabs.platform.sso.oidc.service.XLOidcUserService
import com.xebialabs.platform.sso.oidc.web.{CustomAuthorizationRequestResolver, OidcLogoutSuccessHandler}
import com.xebialabs.xldeploy.auth.oidc.web.XlDeployLoginFormFilter
import com.xebialabs.xldeploy.authentication.oidc.conf.OpenIdConnectConfig._
import com.xebialabs.xldeploy.authentication.oidc.policy.impl.OidcUserProfileCreationPolicy
import com.xebialabs.xlplatform.config.{ConfigLoader, ConfigurationHolder}
import grizzled.slf4j.Logging
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty
import org.springframework.context.annotation.{Bean, ComponentScan, Configuration}
import org.springframework.http.client.{ClientHttpRequestFactory, SimpleClientHttpRequestFactory}
import org.springframework.http.converter.{FormHttpMessageConverter, HttpMessageConverter}
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity
import org.springframework.security.core.AuthenticationException
import org.springframework.security.oauth2.client.endpoint.{DefaultAuthorizationCodeTokenResponseClient, NimbusJwtClientAuthenticationParametersConverter, OAuth2AuthorizationCodeGrantRequest, OAuth2AuthorizationCodeGrantRequestEntityConverter}
import org.springframework.security.oauth2.client.http.OAuth2ErrorResponseErrorHandler
import org.springframework.security.oauth2.client.registration.{ClientRegistration, ClientRegistrationRepository, ClientRegistrations, InMemoryClientRegistrationRepository}
import org.springframework.security.oauth2.client.web.{AuthenticatedPrincipalOAuth2AuthorizedClientRepository, OAuth2AuthorizationRequestRedirectFilter}
import org.springframework.security.oauth2.client.{InMemoryOAuth2AuthorizedClientService, OAuth2AuthorizedClientService}
import org.springframework.security.oauth2.core.http.converter.OAuth2AccessTokenResponseHttpMessageConverter
import org.springframework.security.oauth2.core.{AuthorizationGrantType, ClientAuthenticationMethod, DelegatingOAuth2TokenValidator, OAuth2TokenValidator}
import org.springframework.security.oauth2.jose.jws.{JwsAlgorithm, MacAlgorithm, SignatureAlgorithm}
import org.springframework.security.oauth2.jwt._
import org.springframework.security.oauth2.server.resource.authentication.JwtAuthenticationConverter
import org.springframework.security.web.authentication.session.SessionAuthenticationException
import org.springframework.security.web.authentication.{AuthenticationFailureHandler, LoginUrlAuthenticationEntryPoint, SimpleUrlAuthenticationFailureHandler}
import org.springframework.web.client.RestTemplate
import org.springframework.web.util.UriComponentsBuilder

import java.io.{File, FileInputStream, UnsupportedEncodingException}
import java.net.{InetSocketAddress, Proxy, URI, URLDecoder}
import java.nio.charset.StandardCharsets
import java.security._
import java.security.interfaces.{ECPrivateKey, ECPublicKey, RSAPublicKey}
import java.util
import java.util.UUID
import javax.crypto.spec.SecretKeySpec
import jakarta.servlet.http.{HttpServletRequest, HttpServletResponse}
import org.springframework.security.authentication.{BadCredentialsException, DisabledException}

import scala.collection.mutable.ListBuffer
import scala.jdk.CollectionConverters._
import scala.jdk.FunctionConverters._
import scala.util.{Failure, Success, Try, Using}

object OpenIdConnectConfig {
  val MAC_ALGORITHM_MAPPING : Map[MacAlgorithm, String] = Map(
    MacAlgorithm.HS256 -> "HmacSHA256",
    MacAlgorithm.HS384 -> "HmacSHA384",
    MacAlgorithm.HS512 -> "HmacSHA512"
  )
  val ESA_ALGORITHM_CURVE_MAPPING: Map[JwsAlgorithm, Curve]  = Map(
    SignatureAlgorithm.ES256 -> Curve.P_256,
    SignatureAlgorithm.ES384 -> Curve.P_384,
    SignatureAlgorithm.ES512 -> Curve.P_521
  )

  val ESA_ALGORITHM_MAP: Map[JwsAlgorithm, JWSAlgorithm] = Map(
    SignatureAlgorithm.ES256 -> JWSAlgorithm.ES256,
    SignatureAlgorithm.ES384 -> JWSAlgorithm.ES384,
    SignatureAlgorithm.ES512 -> JWSAlgorithm.ES512
  )

  val RSA_ALGORITHM_MAP: Map[JwsAlgorithm, JWSAlgorithm]  = Map(
    SignatureAlgorithm.RS256 -> JWSAlgorithm.RS256,
    SignatureAlgorithm.RS384 -> JWSAlgorithm.RS384,
    SignatureAlgorithm.RS512 -> JWSAlgorithm.RS512,
    SignatureAlgorithm.PS256 -> JWSAlgorithm.PS256,
    SignatureAlgorithm.PS384 -> JWSAlgorithm.PS384,
    SignatureAlgorithm.PS512 -> JWSAlgorithm.PS512)

}

@ConditionalOnProperty(name = Array("deploy.server.security.auth.provider"), havingValue = "oidc")
@EnableWebSecurity
@Configuration
@ComponentScan(Array("com.xebialabs.xldeploy.auth.oidc.config"))
class OpenIdConnectConfig extends Logging {

  @Bean
  def oidcConfig = new OidcConfig(ConfigLoader.loadWithDynamic(ConfigurationHolder.get()), ServerConfiguration.getInstance())

  // register the login provider in the application context for the AuthResource
  @Bean
  def oidcLoginProvider: LoginProvider = LoginProvider(oidcConfig.loginMethodDescription, "." + oidcConfig.external_login, "./logout", localUrl = false)

  @Bean
  def localLoginProvider: LoginProvider = LoginProvider("XL Deploy local login", "./login", "./logout", localUrl = true)

  @Bean
  def clientRegistrationRepository: InMemoryClientRegistrationRepository = new InMemoryClientRegistrationRepository(this.clientRegistration)

  @Bean
  def claimsToGrantedAuthoritiesPolicy = new DefaultClaimsToGrantedAuthoritiesPolicy(oidcConfig.rolesClaimName)

  @Bean
  def customAuthorizationRequestResolver(clientRegistrationRepository: ClientRegistrationRepository) =
    new CustomAuthorizationRequestResolver(clientRegistrationRepository, OAuth2AuthorizationRequestRedirectFilter.DEFAULT_AUTHORIZATION_REQUEST_BASE_URI, oidcConfig.additionalParameters)

  @Bean
  def authorizedClientService(clientRegistrationRepository: ClientRegistrationRepository) = new InMemoryOAuth2AuthorizedClientService(clientRegistrationRepository)

  @Bean
  def xlOidcUserService(claimsToGrantedAuthoritiesPolicy: ClaimsToGrantedAuthoritiesPolicy) = new XLOidcUserService(claimsToGrantedAuthoritiesPolicy)

  @Bean
  def authorizedClientRepository(authorizedClientService: OAuth2AuthorizedClientService) = new AuthenticatedPrincipalOAuth2AuthorizedClientRepository(authorizedClientService)

  @Bean
  def idTokenDecoderFactory: JwtDecoderFactory[ClientRegistration] = {
    val restTemplate = new RestTemplate
    restTemplate.setRequestFactory(clientHttpRequestFactory)

    val jwsAlgorithm = getJwsAlgorithm(oidcConfig.idTokenJWSAlg, "deploy.security.auth.providers.oidc.idTokenJWSAlg")
    val idTokenDecoderFactory = new CustomOidcIdTokenDecoderFactory
    idTokenDecoderFactory.setJwsAlgorithmResolver((_: ClientRegistration) => jwsAlgorithm)
    idTokenDecoderFactory.setRestOperations(restTemplate)
    idTokenDecoderFactory
  }

  @Bean
  def jwtDecoder: JwtDecoder = {
    val jwsAlgorithm = getJwsAlgorithm(oidcConfig.accessTokenJWSAlg, "deploy.security.auth.providers.oidc.access-token.jwsAlg")
    val jwtDecoder: NimbusJwtDecoder = jwsAlgorithm match {
      case sAlg: SignatureAlgorithm =>
        val jwkSetUri = if (isNotBlank(oidcConfig.accessTokenKeyUri)) {
          oidcConfig.accessTokenKeyUri
        } else {
          this.clientRegistration.getProviderDetails.getJwkSetUri
        }
        NimbusJwtDecoder.withJwkSetUri(jwkSetUri).jwsAlgorithm(sAlg).build
      case mAlg: MacAlgorithm =>
        checkArgument(isNotBlank(oidcConfig.accessTokenSecretKey), "No configuration setting found for key 'deploy.security.auth.providers.oidc.access-token.secretKey'")

        val secretKeySpec: SecretKeySpec = new SecretKeySpec(oidcConfig.accessTokenSecretKey.getBytes(StandardCharsets.UTF_8),
          MAC_ALGORITHM_MAPPING.getOrElse(mAlg, throw new IllegalArgumentException(s"'${mAlg.getName}' is not supported. Ensure you have configured a valid HMAC Algorithm.")))
        NimbusJwtDecoder.withSecretKey(secretKeySpec).macAlgorithm(mAlg).build
      case _ =>
        throw new UnsupportedOidcConfigurationException("Failed to find a Signature Verifier. Ensure you have configured a valid JWS Algorithm.")
    }

    jwtDecoder.setJwtValidator(getJwtOAuth2TokenValidator)
    jwtDecoder
  }

  @Bean
  def jwtAuthenticationConverter: JwtAuthenticationConverter = {
    val jwtAuthenticationConverter = new JwtAuthenticationConverter
    jwtAuthenticationConverter.setPrincipalClaimName(oidcConfig.userNameClaimName)
    jwtAuthenticationConverter.setJwtGrantedAuthoritiesConverter(new GrantedAuthoritiesExtractor(claimsToGrantedAuthoritiesPolicy))
    jwtAuthenticationConverter
  }

  @Bean
  def authorizationCodeTokenResponseClient(clientRegistrationRepository: ClientRegistrationRepository): DefaultAuthorizationCodeTokenResponseClient = {
    val messageConverter = new ListBuffer[HttpMessageConverter[_]]
    messageConverter += new FormHttpMessageConverter
    messageConverter += new OAuth2AccessTokenResponseHttpMessageConverter

    val restTemplate = new RestTemplate(messageConverter.toList.asJava)
    restTemplate.setRequestFactory(clientHttpRequestFactory)
    restTemplate.setErrorHandler(new OAuth2ErrorResponseErrorHandler)

    val authorizationCodeTokenResponseClient = new DefaultAuthorizationCodeTokenResponseClient
    authorizationCodeTokenResponseClient.setRestOperations(restTemplate)

    val clientRegistration = clientRegistrationRepository.findByRegistrationId(oidcConfig.registrationId)
    if (clientRegistration.getClientAuthenticationMethod == ClientAuthenticationMethod.CLIENT_SECRET_JWT ||
      clientRegistration.getClientAuthenticationMethod == ClientAuthenticationMethod.PRIVATE_KEY_JWT) {
      setJwk(clientRegistration)
      val requestEntityConverter = new OAuth2AuthorizationCodeGrantRequestEntityConverter
      requestEntityConverter.addParametersConverter(new NimbusJwtClientAuthenticationParametersConverter[OAuth2AuthorizationCodeGrantRequest](jwkResolver.asJavaFunction))
      authorizationCodeTokenResponseClient.setRequestEntityConverter(requestEntityConverter)
    }

    authorizationCodeTokenResponseClient
  }

  val jwkResolver: ClientRegistration => JWK = (clientRegistration: ClientRegistration) => {
    if (clientRegistration.getClientAuthenticationMethod.equals(ClientAuthenticationMethod.CLIENT_SECRET_JWT) ||
      clientRegistration.getClientAuthenticationMethod.equals(ClientAuthenticationMethod.PRIVATE_KEY_JWT)) {
      oidcConfig.jwk
    } else {
      null
    }
  }

  @Bean
  def oidcLoginFailureHandler: AuthenticationFailureHandler = (request: HttpServletRequest, response: HttpServletResponse, exception: AuthenticationException) => {
    logger.debug(s"xld:auth: oidcLoginFailureHandler called - request URI: ${request.getRequestURI}, context path: ${request.getContextPath}, exception type: ${exception.getClass.getSimpleName}")
    
    exception match {
      case authException @ (_ : BadCredentialsException | _:  DisabledException) =>
        val redirectUrl = request.getContextPath + "/" + XlDeployLoginFormFilter.DEFAULT_LOGIN_PATH + "?" + XlDeployLoginFormFilter.ERROR_PARAMETER_NAME + "=" + authException.getMessage
        logger.debug(s"xld:auth: BadCredentialsException or DisabledException - redirecting to: $redirectUrl")
        response.sendRedirect(redirectUrl)
      case exception: SessionAuthenticationException =>
        val redirectUrl = request.getContextPath + "/" + XlDeployLoginFormFilter.DEFAULT_LOGIN_PATH + "?" + XlDeployLoginFormFilter.ERROR_PARAMETER_NAME + "=" + exception.getMessage
        logger.debug(s"xld:auth: SessionAuthenticationException - redirecting to: $redirectUrl")
        response.sendRedirect(redirectUrl)
      case _ => 
        // include context path so redirects to external login preserve the app context
        val externalRedirect = request.getContextPath + oidcConfig.external_login
        logger.debug(s"xld:auth: Other authentication exception (${exception.getClass.getSimpleName}) - redirecting to external login: $externalRedirect")

        // If this is an XHR or expects JSON (SPA), return a JSON payload with the redirect URL
        val accept = Option(request.getHeader("Accept")).getOrElse("")
        val xRequestedWith = Option(request.getHeader("X-Requested-With")).getOrElse("")
        val isAjax = xRequestedWith.equalsIgnoreCase("XMLHttpRequest") || accept.contains("application/json")

        if (isAjax) {
          val encoded = java.net.URLEncoder.encode(Option(exception.getMessage).getOrElse(""), java.nio.charset.StandardCharsets.UTF_8.name())
          val spaHashRedirect = request.getContextPath + "/#/" + XlDeployLoginFormFilter.DEFAULT_LOGIN_PATH + "?" + XlDeployLoginFormFilter.ERROR_PARAMETER_NAME + "=" + encoded
          logger.debug(s"xld:auth: AJAX request detected - redirecting to SPA hash login: $spaHashRedirect")
          response.sendRedirect(spaHashRedirect)
        } else {
          response.sendRedirect(externalRedirect)
        }
    }
    logger.debug("xld:auth: oidcLoginFailureHandler completed")
  }

  @Bean
  def xldLoginFailureHandler = new SimpleUrlAuthenticationFailureHandler

  @Bean
  def openIdLogoutSuccessHandler(clientRegistrationRepository: ClientRegistrationRepository): OidcLogoutSuccessHandler =
    new OidcLogoutSuccessHandler(clientRegistrationRepository, oidcConfig.postLogoutRedirectUri, XlDeployLoginFormFilter.DEFAULT_LOGIN_PATH)

  @Bean
  def loginUrlAuthenticationEntryPoint = new LoginUrlAuthenticationEntryPoint(oidcConfig.external_login)

  @Bean
  def oidcUserProfileCreationPolicy = new OidcUserProfileCreationPolicy(oidcConfig.emailClaim, oidcConfig.fullNameClaim)

  private def clientRegistration = {
    /*
      Try to create client registration with below two options:

      1. Try to fetch required configuration uris via issuer discovery uri i.e. http://server.com/.well-known/openid-configuration
        see, https://openid.net/specs/openid-connect-discovery-1_0.html#WellKnownRegistry
        User can still override required uris by providing appropriate placeholder configuration as before.

      2. If discovery uri is not present, try to create required configuration by resolving below placeholders. Validation has to be present for this.
        deploy.security.auth.providers.oidc.userAuthorizationUri
        deploy.security.auth.providers.oidc.accessTokenUri
        deploy.security.auth.providers.oidc.keyRetrievalUri
        deploy.security.auth.providers.oidc.logoutUri

      Known Limitation: If issuer uri contains whitespaces, system will not try to fetch data using discovery uri to prevent double encoding.
      see, https://github.com/spring-projects/spring-security/issues/9092
    */

    val clientRegistrationBuilder: ClientRegistration.Builder = getClientRegistrationBuilder
    val authenticationMethod: ClientAuthenticationMethod = getClientAuthMethod

    clientRegistrationBuilder
      .clientId(oidcConfig.clientId)
      .clientAuthenticationMethod(authenticationMethod)
      .authorizationGrantType(AuthorizationGrantType.AUTHORIZATION_CODE)
      .redirectUri(oidcConfig.redirectUri)
      .userInfoUri("")
      .scope(oidcConfig.scopes)
      .userNameAttributeName(oidcConfig.userNameClaimName)

    // clientSecret is required if authentication method is not none or private key jwt
    if (authenticationMethod != ClientAuthenticationMethod.NONE &&
      authenticationMethod != ClientAuthenticationMethod.PRIVATE_KEY_JWT) {
      checkArgument(isNotBlank(oidcConfig.clientSecret), getErrorMessageForPlaceholder("deploy.security.auth.providers.oidc.clientSecret"))
      clientRegistrationBuilder.clientSecret(oidcConfig.clientSecret)
    }
    // validate fields required for authentication method private_key_jwt
    if (authenticationMethod == ClientAuthenticationMethod.PRIVATE_KEY_JWT) {
      if (isBlank(oidcConfig.keyStorePath)) {
        oidcConfig.keyStorePath = oidcConfig.defaultKeyStorePath
        oidcConfig.keyStorePassword = oidcConfig.defaultKeyStorePassword
      }
      checkArgument(isNotBlank(oidcConfig.keyStorePath), getErrorMessageForPlaceholder("deploy.security.auth.providers.oidc.clientAuthJwt.keyStore.path"))
      checkArgument(isNotBlank(oidcConfig.keyAlias), getErrorMessageForPlaceholder("deploy.security.auth.providers.oidc.clientAuthJwt.key.alias"))

    }

    if (isNotBlank(oidcConfig.userAuthorizationUri)) {
      clientRegistrationBuilder.authorizationUri(oidcConfig.userAuthorizationUri)
    }
    if (isNotBlank(oidcConfig.accessTokenUri)) {
      clientRegistrationBuilder.tokenUri(oidcConfig.accessTokenUri)
    }
    if (isNotBlank(oidcConfig.jwks_uri)) {
      clientRegistrationBuilder.jwkSetUri(oidcConfig.jwks_uri)
    }

    val clientRegistration = clientRegistrationBuilder.build

    val configurationMetadata = clientRegistration.getProviderDetails.getConfigurationMetadata
    val updatedConfigurationMetadata = new util.HashMap[String, AnyRef](configurationMetadata)
    updatedConfigurationMetadata.put("rolesClaim", oidcConfig.rolesClaimName)

    if (isNotBlank(oidcConfig.logoutUri)) {
      updatedConfigurationMetadata.put("end_session_endpoint", oidcConfig.logoutUri)
    }

    ClientRegistration.withClientRegistration(clientRegistration).providerConfigurationMetadata(updatedConfigurationMetadata).build()
  }

  private def getClientRegistrationBuilder: ClientRegistration.Builder = {
    try {
      if (oidcConfig.issuer.matches("\\S+")) {
        ClientRegistrations.fromOidcIssuerLocation(oidcConfig.issuer).registrationId(oidcConfig.registrationId)
      } else {
        throw new UnsupportedOidcConfigurationException(s"Whitespace characters in issuer url [${oidcConfig.issuer}] is not supported. Recommendation is to avoid using spaces in URLs, and instead use hyphens to separate words.")
      }
    } catch {
      case re: RuntimeException =>
        re match {
          case _: IllegalArgumentException =>
            logger.warn(s"Unable to resolve Configuration with the provided Issuer of [${oidcConfig.issuer}]")
          case _: UnsupportedOidcConfigurationException =>
            logger.warn(re.getMessage)
        }
        validateOidcConfiguration()
        ClientRegistration.withRegistrationId(oidcConfig.registrationId)
    }
  }

  private def clientHttpRequestFactory: ClientHttpRequestFactory = {
    val requestFactory = new SimpleClientHttpRequestFactory()
    if (isNotBlank(oidcConfig.proxyHost)) {
      requestFactory.setProxy(new Proxy(Proxy.Type.HTTP, new InetSocketAddress(oidcConfig.proxyHost, oidcConfig.proxyPort)))
    }
    requestFactory
  }

  private def getJwsAlgorithm(jwsAlgorithm: String, propertyName: String): JwsAlgorithm = {
    val jwsAlgorithms = SignatureAlgorithm.values().map(v => (v.getName, v)).toMap ++ MacAlgorithm.values().map(v => (v.getName, v)).toMap

    jwsAlgorithms.getOrElse(jwsAlgorithm,
      throw new IllegalArgumentException(s"$propertyName value [$jwsAlgorithm] is not supported. Ensure you have configured a valid JWS Algorithm."))
  }

  private def getMacAlgorithm(macAlgorithm: String, propertyName: String): MacAlgorithm = {
    val macAlgorithms = MacAlgorithm.values().map(v => (v.getName, v)).toMap

    macAlgorithms.getOrElse(macAlgorithm,
      throw new IllegalArgumentException(s"$propertyName value [$macAlgorithm] is not supported. Ensure you have configured a valid HMAC Algorithm."))
  }

  private def getSignatureAlgorithm(signatureAlgorithm: String, propertyName: String): SignatureAlgorithm  = {
    val signatureAlgorithms = SignatureAlgorithm.values().map(v => (v.getName, v)).toMap

    signatureAlgorithms.getOrElse(signatureAlgorithm,
      throw new IllegalArgumentException(s"$propertyName value [$signatureAlgorithm] is not supported. Ensure you have configured a valid Signature Algorithm."))
  }

  private def getClientAuthMethod: ClientAuthenticationMethod = {
    val clientAuthMethods = Map(
      "client_secret_basic" -> ClientAuthenticationMethod.CLIENT_SECRET_BASIC,
      "client_secret_post" -> ClientAuthenticationMethod.CLIENT_SECRET_POST,
      "client_secret_jwt" -> ClientAuthenticationMethod.CLIENT_SECRET_JWT,
      "private_key_jwt" -> ClientAuthenticationMethod.PRIVATE_KEY_JWT,
      "none" -> ClientAuthenticationMethod.NONE
    )

    clientAuthMethods.getOrElse(oidcConfig.clientAuthMethod.toLowerCase,
      throw new IllegalArgumentException(s"clientAuthMethod value [${oidcConfig.clientAuthMethod} is not supported. Ensure you have configured a valid client authentication method.]"))
  }

  private def getJwtOAuth2TokenValidator: OAuth2TokenValidator[Jwt] = {
    val validators = new util.ArrayList[OAuth2TokenValidator[Jwt]]
    validators.add(new JwtTimestampValidator())
    validators.add(new JwtIssuerValidator(if (isNotBlank(oidcConfig.accessTokenIssuer)) oidcConfig.accessTokenIssuer else oidcConfig.issuer))
    validators.add(new JwtClaimValidator[util.List[String]](JwtClaimNames.AUD,
      (aud: util.List[String]) => aud.contains(if (isNotBlank(oidcConfig.accessTokenAudience)) oidcConfig.accessTokenAudience else oidcConfig.clientId)))
    new DelegatingOAuth2TokenValidator[Jwt](validators)
  }

  private def validateOidcConfiguration(): Unit = {
    checkArgument(isNotBlank(oidcConfig.userAuthorizationUri), getErrorMessageForPlaceholder("deploy.security.auth.providers.oidc.userAuthorizationUri"))
    checkArgument(isNotBlank(oidcConfig.accessTokenUri), getErrorMessageForPlaceholder("deploy.security.auth.providers.oidc.accessTokenUri"))
    checkArgument(isNotBlank(oidcConfig.jwks_uri), getErrorMessageForPlaceholder("deploy.security.auth.providers.oidc.keyRetrievalUri"))
    checkArgument(isNotBlank(oidcConfig.logoutUri), getErrorMessageForPlaceholder("deploy.security.auth.providers.oidc.logoutUri"))
  }

  private def getErrorMessageForPlaceholder(placeholder: String) = s"No configuration setting found for key '$placeholder'"

  private def setJwk(clientRegistration: ClientRegistration): Unit = {
    if (clientRegistration.getClientAuthenticationMethod.equals(ClientAuthenticationMethod.CLIENT_SECRET_JWT)) {
      val jwsAlgorithm = getMacAlgorithm(oidcConfig.clientAuthJWSAlg,
        "deploy.security.auth.providers.oidc.clientAuthJwt.jwsAlg")
      jwsAlgorithm match {
        case mAlg: MacAlgorithm =>
          val secretKey: SecretKeySpec = new SecretKeySpec(
            clientRegistration.getClientSecret.getBytes(StandardCharsets.UTF_8),
            MAC_ALGORITHM_MAPPING.getOrElse(mAlg, throw new IllegalArgumentException(s"'${mAlg.getName}' is not " +
              s"supported. Ensure you have configured a valid HMAC Algorithm."))
          )
          oidcConfig.jwk = new OctetSequenceKey.Builder(secretKey).algorithm(new Algorithm(mAlg.getName))
            .keyUse(KeyUse.SIGNATURE).keyID(UUID.randomUUID.toString).build
        case _ =>
          throw new UnsupportedOidcConfigurationException("Failed to find a Signature Verifier. Ensure you have " +
            "configured a valid JWS Algorithm.")
      }
    }
    else if (clientRegistration.getClientAuthenticationMethod.equals(ClientAuthenticationMethod.PRIVATE_KEY_JWT)) {
      val jwsAlgorithm = getSignatureAlgorithm(oidcConfig.clientAuthJWSAlg,
        "deploy.security.auth.providers.oidc.clientAuthJwt.jwsAlg")
      jwsAlgorithm match {
        case _: SignatureAlgorithm =>
          val keyPair: KeyPair = getKeyPair
          if (RSA_ALGORITHM_MAP.contains(jwsAlgorithm)) oidcConfig.jwk = getRSAKey(keyPair,
            RSA_ALGORITHM_MAP(jwsAlgorithm))
          else if (ESA_ALGORITHM_MAP.contains(jwsAlgorithm)) oidcConfig.jwk = getESAKey(keyPair,
            ESA_ALGORITHM_MAP(jwsAlgorithm), ESA_ALGORITHM_CURVE_MAPPING(jwsAlgorithm))
          else throw new UnsupportedOidcConfigurationException("Ensure you have configured a valid JWS Algorithm.")
        case _ =>
          throw new UnsupportedOidcConfigurationException("Failed to find a Signature Verifier. Ensure you have " +
            "configured a valid JWS Algorithm.")
      }
    } else {
      oidcConfig.jwk = null
    }
  }

  private def getKeyPair = {
    val file = new File(oidcConfig.keyStorePath)
    if (!file.exists) {
      throw new IllegalArgumentException(s"Cannot find keystore [${oidcConfig.keyStorePath}]")
    }
    if (isBlank(oidcConfig.keyStorePassword)) {
      logger.warn(s"The keystore [${oidcConfig.keyStorePath}] is not protected by a password. " +
        "It is recommended to secure it using a password.")
    }
    val tryKeyPair: Try[KeyPair] = Using(new FileInputStream(file)) { in =>
      val kst: String = if (isNotBlank(oidcConfig.keyStoreType)) oidcConfig.keyStoreType else KeyStore.getDefaultType
      val keystore = KeyStore.getInstance(kst)
      val ksp = if (isNotBlank(oidcConfig.keyStorePassword)) oidcConfig.keyStorePassword.toCharArray else null
      keystore.load(in, ksp)

      val kp = if (isNotBlank(oidcConfig.keyPassword)) oidcConfig.keyPassword.toCharArray else null
      val privateKey = keystore.getKey(oidcConfig.keyAlias, kp).asInstanceOf[PrivateKey]
      val cert = keystore.getCertificate(oidcConfig.keyAlias)
      val publicKey = cert.getPublicKey
      new KeyPair(publicKey, privateKey)
    }
    tryKeyPair match {
      case Success(value) => value
      case Failure(exception) => throw new DeployitException(exception)
    }
  }

  private def getRSAKey(keyPair: KeyPair, algorithm: Algorithm) = {
    val builder = new RSAKey.Builder(keyPair.getPublic.asInstanceOf[RSAPublicKey]).privateKey(keyPair.getPrivate)
      .algorithm(algorithm).keyUse(KeyUse.SIGNATURE)
    if (isNotBlank(oidcConfig.tokenKeyId)) {
      builder.keyID(oidcConfig.tokenKeyId)
    }
    else {
      logger.debug("Building RSAKey without token k-id. Token key Id not provided in configuration.")
    }
    builder.build
  }

  private def getESAKey(keyPair: KeyPair, algorithm: Algorithm, curve: Curve) = {
    val builder = new ECKey.Builder(curve, keyPair.getPublic.asInstanceOf[ECPublicKey])
      .privateKey(keyPair.getPrivate.asInstanceOf[ECPrivateKey]).algorithm(algorithm).keyUse(KeyUse.SIGNATURE)
    if (isNotBlank(oidcConfig.tokenKeyId)) {
      builder.keyID(oidcConfig.tokenKeyId)
    }
    else {
      logger.debug("Building ECKey without token k-id. Token key Id not provided in configuration.")
    }
    builder.build
  }

}

class OidcConfig(config: Config, serverConfiguration: ServerConfiguration) {
  val registrationId: String = "xl-deploy"
  val clientId: String = config.getString("deploy.security.auth.providers.oidc.clientId")
  val clientSecret: String = getStringProperty(config, "deploy.security.auth.providers.oidc.clientSecret", "")
  val clientAuthMethod: String = getStringProperty(config, "deploy.security.auth.providers.oidc.clientAuthMethod", "client_secret_post")
  val clientAuthJWSAlg: String = getStringProperty(config, "deploy.security.auth.providers.oidc.clientAuthJwt.jwsAlg", MacAlgorithm.HS256.getName)
  val tokenKeyId: String = getStringProperty(config, "deploy.security.auth.providers.oidc.clientAuthJwt.tokenKeyId", "")
  var keyStorePath: String = getStringProperty(config, "deploy.security.auth.providers.oidc.clientAuthJwt.keyStore.path", "")
  var keyStorePassword: String = getStringProperty(config, "deploy.security.auth.providers.oidc.clientAuthJwt.keyStore.password", "")
  var keyStoreType: String = getStringProperty(config, "deploy.security.auth.providers.oidc.clientAuthJwt.keyStore.type", "")
  val keyAlias: String = getStringProperty(config, "deploy.security.auth.providers.oidc.clientAuthJwt.key.alias", "")
  val keyPassword: String = getStringProperty(config, "deploy.security.auth.providers.oidc.clientAuthJwt.key.password", "")
  val defaultKeyStorePath: String = serverConfiguration.getKeyStorePath
  val defaultKeyStorePassword: String = serverConfiguration.getKeyStorePassword

  val issuer: String = getDecodedUriString(config.getString("deploy.security.auth.providers.oidc.issuer"))
  val jwks_uri: String = getEncodedUriString(getStringProperty(config, "deploy.security.auth.providers.oidc.keyRetrievalUri", ""))
  val accessTokenUri: String = getDecodedUriString(getStringProperty(config, "deploy.security.auth.providers.oidc.accessTokenUri", ""))
  val userAuthorizationUri: String = getEncodedUriString(getStringProperty(config, "deploy.security.auth.providers.oidc.userAuthorizationUri", ""))
  val logoutUri: String = getDecodedUriString(getStringProperty(config, "deploy.security.auth.providers.oidc.logoutUri", ""))

  val redirectUri: String = getEncodedUriString(getStringProperty(config, "deploy.security.auth.providers.oidc.redirectUri", s"${serverConfiguration.getServerUrl}login/external-login"))
  val postLogoutRedirectUri: String = getStringProperty(config, "deploy.security.auth.providers.oidc.postLogoutRedirectUri", redirectUri)

  val rolesClaimName: String = config.getString("deploy.security.auth.providers.oidc.rolesClaimName")
  val userNameClaimName: String = config.getString("deploy.security.auth.providers.oidc.userNameClaimName")
  val idTokenJWSAlg: String = getStringProperty(config, "deploy.security.auth.providers.oidc.idTokenJWSAlg", SignatureAlgorithm.RS256.getName)
  val fullNameClaim: String = getStringProperty(config, "deploy.security.auth.providers.oidc.fullNameClaim", "")
  val emailClaim: String = getStringProperty(config, "deploy.security.auth.providers.oidc.emailClaim", "")

  val additionalParameters: util.Map[String, Object] = getMapProperty(config, "deploy.security.auth.providers.oidc.additionalParameters", new util.HashMap[String, Object]())

  val accessTokenIssuer: String = getStringProperty(config, "deploy.security.auth.providers.oidc.access-token.issuer", "")
  val accessTokenAudience: String = getStringProperty(config, "deploy.security.auth.providers.oidc.access-token.audience", "")
  val accessTokenKeyUri: String = getStringProperty(config, "deploy.security.auth.providers.oidc.access-token.keyRetrievalUri", "")
  val accessTokenJWSAlg: String = getStringProperty(config, "deploy.security.auth.providers.oidc.access-token.jwsAlg", SignatureAlgorithm.RS256.getName)
  val accessTokenSecretKey: String = getStringProperty(config, "deploy.security.auth.providers.oidc.access-token.secretKey", "")

  val external_login: String = s"${OAuth2AuthorizationRequestRedirectFilter.DEFAULT_AUTHORIZATION_REQUEST_BASE_URI}/$registrationId"
  val loginMethodDescription: String = getStringProperty(config, "deploy.security.auth.providers.oidc.loginMethodDescription", "External login (OpenID Connect)")
  val proxyHost: String = getStringProperty(config, "deploy.security.auth.providers.oidc.proxyHost", null)
  var proxyPort: Integer = getIntProperty(config, "deploy.security.auth.providers.oidc.proxyPort", null)

  var jwk: JWK = _

  var scopes: util.List[String] = getStringListProperty(config, "deploy.security.auth.providers.oidc.scopes",
    new util.ArrayList[String]() {
      add("openid")
    })

  private def getStringProperty(config: Config, name: String, defaultValue: String): String = try
    config.getString(name)
  catch {
    case _: ConfigException.Missing => defaultValue
  }

  private def getIntProperty(config: Config, name: String, defaultValue: Integer): Integer = try
    config.getInt(name)
  catch {
    case _: ConfigException.Missing => defaultValue
  }

  private def getEncodedUriString(uri: String): String = try
    URI.create(uri).toASCIIString
  catch {
    case _: IllegalArgumentException => UriComponentsBuilder.fromUriString(uri).build.toUri.toASCIIString
  }

  private def getDecodedUriString(uri: String) = try
    URLDecoder.decode(uri, StandardCharsets.UTF_8.name)
  catch {
    case _: UnsupportedEncodingException => throw new IllegalArgumentException(String.format("Cannot decode value [%s] with 'utf-8' charset", uri))
  }

  private def getStringListProperty(config: Config, name: String, defaultValue: util.List[String]): util.List[String] = {
    try config.getStringList(name)
    catch {
      case _: ConfigException.Missing => defaultValue
    }
  }

  private def getMapProperty(config: Config, name: String, defaultValue: util.Map[String, Object]): util.Map[String, Object] = {
    try config.getObject(name).unwrapped()
    catch {
      case _: ConfigException.Missing => defaultValue
    }
  }
}
