package com.xebialabs.xldeploy.authentication.oidc.conf

import ai.digital.configuration.central.deploy.ClientProperties
import com.xebialabs.deployit.filter.CsrfHeaderFilter
import com.xebialabs.xldeploy.auth.oidc.web.XlDeployLoginFormFilter
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty
import org.springframework.context.annotation.{Bean, Configuration}
import org.springframework.core.annotation.Order
import org.springframework.security.config.annotation.web.builders.HttpSecurity
import org.springframework.security.config.annotation.web.configuration.{EnableWebSecurity, WebSecurityCustomizer}
import org.springframework.security.web.SecurityFilterChain
import org.springframework.security.web.authentication.UsernamePasswordAuthenticationFilter
import org.springframework.security.web.authentication.logout.{CookieClearingLogoutHandler, LogoutFilter}
import org.springframework.security.web.csrf.{CsrfFilter, CsrfLogoutHandler, CsrfTokenRequestAttributeHandler}
import org.springframework.security.web.util.matcher.AntPathRequestMatcher.antMatcher

@ConditionalOnProperty(name = Array("deploy.server.security.auth.provider"), havingValue = "oidc")
@Configuration
@EnableWebSecurity
class DeployCsrfSecurityConfig extends DeploySecurityConfig {

  @Autowired
  var clientConfiguration: ClientProperties = _

  private val prefix = "xldeploy"

  @Bean
  @Order(1)
  @throws[Exception]
  def securityFilterChainCsrfSecurityConfig(http: HttpSecurity): SecurityFilterChain = {
    http.securityMatchers(
      securityMatcher =>
        securityMatcher
          .requestMatchers(antMatcher(s"/$prefix/**"))
    )
    configureSecurity(http, prefix = prefix)

    val requestHandler = new CsrfTokenRequestAttributeHandler
    // set the name of the attribute the CsrfToken will be populated on
    requestHandler.setCsrfRequestAttributeName(null)

    http
      .addFilterAfter(new CsrfHeaderFilter(clientConfiguration), classOf[CsrfFilter])
      .addFilterBefore(logoutFilter, classOf[LogoutFilter])
      .addFilterBefore(csrfLoginFormFilter(), classOf[UsernamePasswordAuthenticationFilter])
      .csrf(csrf =>
        csrf
          .csrfTokenRepository(tokenRepository)
          .csrfTokenRequestHandler(requestHandler))
      .build()
  }

  private def csrfLoginFormFilter(): XlDeployLoginFormFilter =
    buildLoginFormFilter(authenticationManager, csrfAuthenticationStrategy)

  private def logoutFilter: LogoutFilter = {
    val logoutFilter = new LogoutFilter(
      openIdLogoutSuccessHandler,
      new CsrfLogoutHandler(tokenRepository),
      new CookieClearingLogoutHandler("cookiesToClear", "XSRF-TOKEN")
    )
    logoutFilter.setFilterProcessesUrl("/xldeploy/logout")
    logoutFilter
  }

  @Bean
  def webCsrfSecurityCustomizer: WebSecurityCustomizer = {
    web => configureSecurity(web, prefix = prefix)
  }
}
