package com.xebialabs.xldeploy.authentication.oidc.conf

import ai.digital.config.ServerConfigurationHelper
import ai.digital.configuration.central.deploy.{ClientProperties, ServerSideProperties}
import com.xebialabs.deployit.core.auth.LoginMetadataService
import com.xebialabs.deployit.engine.api.distribution.TaskExecutionWorkerRepository
import com.xebialabs.deployit.plumbing.authentication.WithoutRedirectLoginSuccessHandler
import com.xebialabs.deployit.security.RoleService
import com.xebialabs.deployit.security.authentication.{BasicAuthWithRememberMeFilter, RememberMeAuthenticationProvider}
import com.xebialabs.deployit.taskexecution.security.{TaskWorkerAuthenticationFilter, TaskWorkerAuthenticationProvider}
import com.xebialabs.deployit.{LicenseExpiryCheckFilter, LogbackAccessSecurityAttributesSaveFilter}
import com.xebialabs.license.LicenseValidationFilter
import com.xebialabs.license.service.LicenseService
import com.xebialabs.platform.sso.oidc.service.XLOidcUserService
import com.xebialabs.platform.sso.oidc.web.{CustomAuthorizationRequestResolver, OidcLogoutSuccessHandler}
import com.xebialabs.xldeploy.auth.BasicAuthOverridingHttpSessionSecurityContextRepository
import com.xebialabs.xldeploy.auth.config.{InMemoryConfigurer, LoginAuthorizationManager}
import com.xebialabs.xldeploy.auth.oidc.web.XlDeployLoginFormFilter
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.{Autowired, Qualifier, Value}
import org.springframework.context.annotation.{Bean, Lazy}
import org.springframework.context.{ApplicationContext, ApplicationEventPublisher}
import org.springframework.http.HttpMethod
import org.springframework.security.authentication.{AuthenticationManager, AuthenticationProvider, DefaultAuthenticationEventPublisher, ProviderManager}
import org.springframework.security.config.annotation.web.builders.{HttpSecurity, WebSecurity}
import org.springframework.security.config.http.SessionCreationPolicy
import org.springframework.security.core.session.SessionRegistry
import org.springframework.security.ldap.authentication.LdapAuthenticationProvider
import org.springframework.security.oauth2.client.OAuth2AuthorizedClientService
import org.springframework.security.oauth2.client.endpoint.DefaultAuthorizationCodeTokenResponseClient
import org.springframework.security.oauth2.client.registration.ClientRegistrationRepository
import org.springframework.security.oauth2.client.web.OAuth2AuthorizedClientRepository
import org.springframework.security.oauth2.jwt.JwtDecoder
import org.springframework.security.web.access.AccessDeniedHandler
import org.springframework.security.web.access.channel.ChannelProcessingFilter
import org.springframework.security.web.access.intercept.RequestAuthorizationContext
import org.springframework.security.web.authentication.logout.HttpStatusReturningLogoutSuccessHandler
import org.springframework.security.web.authentication.session.{ChangeSessionIdAuthenticationStrategy, CompositeSessionAuthenticationStrategy, ConcurrentSessionControlAuthenticationStrategy, SessionAuthenticationStrategy}
import org.springframework.security.web.authentication.{AuthenticationFailureHandler, DelegatingAuthenticationEntryPoint, RememberMeServices, UsernamePasswordAuthenticationFilter}
import org.springframework.security.web.context.request.async.WebAsyncManagerIntegrationFilter
import org.springframework.security.web.csrf.{CsrfAuthenticationStrategy, HttpSessionCsrfTokenRepository}
import org.springframework.security.web.firewall.StrictHttpFirewall
import org.springframework.security.web.savedrequest.NullRequestCache
import org.springframework.security.web.util.matcher.AntPathRequestMatcher.antMatcher
import org.springframework.web.filter.RequestContextFilter

import java.util
import java.util.{LinkedList => LList, List => JList}
import scala.jdk.CollectionConverters._

abstract class DeploySecurityConfig {

  private val logger = LoggerFactory.getLogger(classOf[DeploySecurityConfig])

  @Value("${deploy.server.license.days-before-warning:5}")
  var daysBeforeWarning: Int = _

  @Autowired
  var openIdLogoutSuccessHandler: OidcLogoutSuccessHandler = _

  @Autowired
  var applicationContext: ApplicationContext = _

  @Autowired
  var tokenRepository: HttpSessionCsrfTokenRepository = _

  @Autowired
  var loginMetadataService: LoginMetadataService = _

  @Autowired
  var roleService: RoleService = _

  @Autowired
  var accessDeniedHandler: AccessDeniedHandler = _

  @Autowired
  @Qualifier("rememberMeKey")
  var rememberMeKey: String = _

  @Autowired
  var rememberMeServices: RememberMeServices = _

  @Autowired
  var withoutRedirectLogoutSuccessHandler: HttpStatusReturningLogoutSuccessHandler = _

  @Autowired
  var workerRepository: TaskExecutionWorkerRepository = _

  @Autowired
  var licenseService: LicenseService = _

  @Autowired
  var taskWorkerAuthenticationProvider: TaskWorkerAuthenticationProvider = _

  @Autowired(required = false)
  var ldapAuthenticationProvider: java.util.List[LdapAuthenticationProvider] = _

  @Lazy
  @Autowired
  @Qualifier("xlAuthenticationProvider")
  var xlAuthenticationProvider: AuthenticationProvider = _

  @Autowired
  var delegatingAuthenticationEntryPoint: DelegatingAuthenticationEntryPoint = _

  @Autowired
  var delegatingSecurityContextRepository: BasicAuthOverridingHttpSessionSecurityContextRepository = _

  @Autowired
  var authorizedClientRepository: OAuth2AuthorizedClientRepository = _

  @Autowired
  var clientRegistrationRepository: ClientRegistrationRepository = _

  @Autowired
  var authorizedClientService: OAuth2AuthorizedClientService = _

  @Autowired
  var xlOidcUserService: XLOidcUserService = _

  @Autowired
  var authorizationCodeTokenResponseClient: DefaultAuthorizationCodeTokenResponseClient = _

  @Autowired
  var idTokenDecoderFactory: JwtDecoder = _

  @Autowired
  @Qualifier("oidcLoginFailureHandler")
  var oidcLoginFailureHandler: AuthenticationFailureHandler = _

  @Autowired
  var customAuthorizationRequestResolver: CustomAuthorizationRequestResolver = _

  @Autowired
  @Qualifier("authenticationFailureHandler")
  var authenticationFailureHandler: AuthenticationFailureHandler = _

  @Autowired
  var serverSideConfiguration: ServerSideProperties = _

  @Autowired
  var sessionRegistry: SessionRegistry = _

  @Autowired
  var clientProperties: ClientProperties = _

  @Autowired
  var authenticationManager: AuthenticationManager = _

  @Bean
  @throws[Exception]
  def authenticationManager(applicationEventPublisher: ApplicationEventPublisher): AuthenticationManager = {
    val authenticationProviders = new util.ArrayList[AuthenticationProvider]()
    authenticationProviders.add(taskWorkerAuthenticationProvider)
    authenticationProviders.add(new RememberMeAuthenticationProvider())
    authenticationProviders.add(xlAuthenticationProvider)
    if (ldapAuthenticationProvider != null) {
      for (provider <- ldapAuthenticationProvider.asScala) {
        authenticationProviders.add(provider)
      }
    }
    authenticationProviders.add(InMemoryConfigurer.centralConfigDaoProvider)
    val manager = new ProviderManager(authenticationProviders)
    manager.setAuthenticationEventPublisher(
      new DefaultAuthenticationEventPublisher(applicationEventPublisher)
    )
    manager
  }

  def configureSecurity(web: WebSecurity, prefix: String): Unit = {
    val firewall = new StrictHttpFirewall()
    firewall.setAllowUrlEncodedSlash(true)
    firewall.setAllowUrlEncodedPercent(true)

    if (serverSideConfiguration.http.allowedHosts.enabled) {
      firewall.setAllowedHostnames(serverSideConfiguration.http.allowedHosts.hostnames.contains)
    }
    web.httpFirewall(firewall)

    web.ignoring().requestMatchers(antMatcher("/productregistration"))
    web.ignoring().requestMatchers(antMatcher(s"/$prefix/internal/configuration/properties"))
    web.ignoring().requestMatchers(antMatcher(HttpMethod.GET, s"/$prefix/settings/general"))
    web.ignoring().requestMatchers(antMatcher(s"/$prefix/ha/health"))
    if (clientProperties.ignoreAuthCheckForServerStateApi)
      web.ignoring().requestMatchers(antMatcher(s"/$prefix/server/state"))
    web.ignoring().requestMatchers(antMatcher("/**.js**"))
    web.ignoring().requestMatchers(antMatcher("/**.html"))
    web.ignoring().requestMatchers(antMatcher("/**.css"))
    web.ignoring().requestMatchers(antMatcher("/favicon.ico"))
    web.ignoring().requestMatchers(antMatcher("/fonts/**"))
    web.ignoring().requestMatchers(antMatcher("/icons/**"))
    web.ignoring().requestMatchers(antMatcher("/images/**"))
  }

  def licenseValidationFilter: LicenseValidationFilter = {
    val filter = new LicenseValidationFilter()
    filter.setLicenseService(licenseService)
    filter
  }

  private def defaultOidcMustacheTemplateSettings: util.Map[String, String] = {
    val scope = new util.HashMap[String, String]
    scope.put("productLogo", "./images/xl-deploy-logo.svg")
    scope.put("productColor", "#0099cc")
    scope.put("productIcon", "./favicon.ico")
    scope.put("productName", "XL Deploy")
    scope
  }

  protected def buildLoginFormFilter(authenticationManager: AuthenticationManager,
                                   sessionAuthenticationStrategy: SessionAuthenticationStrategy): XlDeployLoginFormFilter = {
    val filter = new XlDeployLoginFormFilter(defaultOidcMustacheTemplateSettings)
    filter.setAuthenticationManager(authenticationManager)
    filter.setSessionAuthenticationStrategy(sessionAuthenticationStrategy)
    filter.setAuthenticationFailureHandler(authenticationFailureHandler)
    filter
  }

  protected def csrfAuthenticationStrategy: CompositeSessionAuthenticationStrategy = {
    val delegateStrategies: JList[SessionAuthenticationStrategy] = new LList[SessionAuthenticationStrategy]()
    delegateStrategies.add(new ChangeSessionIdAuthenticationStrategy())
    delegateStrategies.add(new CsrfAuthenticationStrategy(tokenRepository))
    delegateStrategies.add(concurrentSessionControlAuthenticationStrategy)
    new CompositeSessionAuthenticationStrategy(delegateStrategies)
  }

  protected def noCsrfAuthenticationStrategy: CompositeSessionAuthenticationStrategy = {
    val delegateStrategies: JList[SessionAuthenticationStrategy] = new LList[SessionAuthenticationStrategy]()
    delegateStrategies.add(new ChangeSessionIdAuthenticationStrategy())
    delegateStrategies.add(concurrentSessionControlAuthenticationStrategy)
    new CompositeSessionAuthenticationStrategy(delegateStrategies)
  }

  private def concurrentSessionControlAuthenticationStrategy: ConcurrentSessionControlAuthenticationStrategy = {
    val authenticationStrategy = new ConcurrentSessionControlAuthenticationStrategy(sessionRegistry)
    authenticationStrategy.setMaximumSessions(serverSideConfiguration.session.maximumSessions)
    authenticationStrategy.setExceptionIfMaximumExceeded(serverSideConfiguration.session.exceptionIfMaximumExceeded)
    authenticationStrategy
  }

  def configureSecurity(http: HttpSecurity, prefix: String): Unit = {
    val loginAuthorizationManager = new LoginAuthorizationManager[RequestAuthorizationContext]()
    loginAuthorizationManager.setWhitelistUrls(Set("/deployit/auth", "/xldeploy/auth").asJava)
    http
      .sessionManagement(smc =>
        smc
          .sessionCreationPolicy(SessionCreationPolicy.NEVER)
          .maximumSessions(serverSideConfiguration.session.maximumSessions)
          .maxSessionsPreventsLogin(serverSideConfiguration.session.exceptionIfMaximumExceeded)
      )
      .securityContext(sc =>
        sc
          .securityContextRepository(delegatingSecurityContextRepository)
          .requireExplicitSave(false)
          .configure(http)
      )
      .addFilterBefore(licenseValidationFilter, classOf[WebAsyncManagerIntegrationFilter])
      .addFilterBefore(new RequestContextFilter(), classOf[ChannelProcessingFilter])
      .addFilterAfter(new TaskWorkerAuthenticationFilter(authenticationManager, workerRepository), classOf[UsernamePasswordAuthenticationFilter])
      .addFilterBefore(new LogbackAccessSecurityAttributesSaveFilter(), classOf[TaskWorkerAuthenticationFilter])
      .addFilterAfter(new BasicAuthWithRememberMeFilter(authenticationManager, delegatingAuthenticationEntryPoint), classOf[TaskWorkerAuthenticationFilter])
      .addFilterAfter(new LicenseExpiryCheckFilter(licenseService, daysBeforeWarning), classOf[TaskWorkerAuthenticationFilter])
      .exceptionHandling(eh =>
        eh
          .authenticationEntryPoint(delegatingAuthenticationEntryPoint)
          .accessDeniedHandler(accessDeniedHandler))
      .formLogin(fl =>
        fl
        .failureHandler(authenticationFailureHandler)
        .successHandler(new WithoutRedirectLoginSuccessHandler())
        .loginPage("/login/external-login")
      )
      .rememberMe(rm =>
        rm
        .key(rememberMeKey)
        .rememberMeServices(rememberMeServices)
      )
      .logout( lc =>
        lc
          .logoutSuccessHandler(openIdLogoutSuccessHandler)
      )
      .requestCache(rcc =>
        rcc
          .requestCache(new NullRequestCache())
      )
      .authorizeHttpRequests(arc => {
        arc
          .requestMatchers(antMatcher(s"/$prefix/auth")).permitAll()
          .requestMatchers(antMatcher("/centralConfiguration/**")).hasRole(ServerConfigurationHelper.ConfigRole)
          .requestMatchers(antMatcher("/actuator/**")).hasRole("ADMIN")
          .requestMatchers(antMatcher("/v1/roles/**")).denyAll()
          .requestMatchers(antMatcher(s"/$prefix/**")).access(loginAuthorizationManager)
          .requestMatchers(antMatcher(s"/$prefix/**")).fullyAuthenticated()
          .requestMatchers(antMatcher("/ciExplorerDist/**")).permitAll()
          .requestMatchers(antMatcher("/stitchWorkbenchDist/**")).permitAll()
          .requestMatchers(antMatcher("/pluginManagerDist/**")).permitAll()
          .requestMatchers(antMatcher("/dist/**")).permitAll()
          .requestMatchers(antMatcher(HttpMethod.GET,"/")).permitAll()
          .requestMatchers(antMatcher("/**")).fullyAuthenticated()
      })

    http
      .oauth2Login(oauth =>
        oauth
          .failureHandler(oidcLoginFailureHandler)
          .authorizedClientService(authorizedClientService)
          .authorizedClientRepository(authorizedClientRepository)
          .authorizationEndpoint(
            authend => authend.authorizationRequestResolver(customAuthorizationRequestResolver)
          )
          .clientRegistrationRepository(clientRegistrationRepository)
          .loginProcessingUrl("/login/external-login")
          .userInfoEndpoint(userInfoEndpt =>
             userInfoEndpt
               .oidcUserService(xlOidcUserService)
               .userAuthoritiesMapper(DeployOidcGrantedAuthoritiesMapper.grantedAuthoritiesMapper)
         )
          .tokenEndpoint(token =>
            token
              .accessTokenResponseClient(authorizationCodeTokenResponseClient)
          )
      )

    http
      .oauth2Client(oauthClient =>
        oauthClient
          .authorizedClientService(authorizedClientService)
          .authorizedClientRepository(authorizedClientRepository)
          .clientRegistrationRepository(clientRegistrationRepository)
          .authorizationCodeGrant(
            authCodeGra =>
              authCodeGra
                .authorizationRequestResolver(customAuthorizationRequestResolver)
                .accessTokenResponseClient(authorizationCodeTokenResponseClient)
          )
      )
    http
      .oauth2ResourceServer(oauth2 =>
       oauth2.jwt(
        jwt => jwt.decoder(idTokenDecoderFactory)
      )
      )

    logger.info("Starting url prefix '{}' with oidc security config", prefix)
  }
}
