package com.xebialabs.xldeploy.authentication.oidc.conf

import com.xebialabs.xldeploy.auth.oidc.web.XlDeployLoginFormFilter
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty
import org.springframework.context.annotation.Configuration
import org.springframework.core.annotation.Order
import org.springframework.security.config.annotation.web.builders.{HttpSecurity, WebSecurity}
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity
import org.springframework.security.web.authentication.UsernamePasswordAuthenticationFilter
import org.springframework.security.web.authentication.logout.{CookieClearingLogoutHandler, LogoutFilter}
import org.springframework.security.web.csrf.CsrfLogoutHandler

@ConditionalOnProperty(name = Array("deploy.server.security.auth.provider"), havingValue = "oidc")
@Configuration
@EnableWebSecurity
@Order(org.springframework.core.Ordered.LOWEST_PRECEDENCE)
class DeployNoCsrfSecurityConfig extends DeploySecurityConfig {

  private val prefix = "deployit"

  @throws[Exception]
  override protected def configure(http: HttpSecurity): Unit = {
    configureSecurity(http, prefix = prefix)

    http
      .headers()
      .frameOptions()
      .sameOrigin().and()
      .authorizeRequests()
      .antMatchers("/api/**").fullyAuthenticated().and()
      .csrf().disable()
      .addFilterBefore(publicLogoutFilter, classOf[LogoutFilter])
      .addFilterBefore(defaultLoginFormFilter(), classOf[UsernamePasswordAuthenticationFilter])
  }

  override def configure(web: WebSecurity): Unit = {
    configureSecurity(web, prefix = prefix)
  }

  private def defaultLoginFormFilter(): XlDeployLoginFormFilter =
    buildLoginFormFilter(authenticationManager, noCsrfAuthenticationStrategy)

  private def publicLogoutFilter: LogoutFilter = {
    val logoutFilter = new LogoutFilter(
      openIdLogoutSuccessHandler,
      new CsrfLogoutHandler(tokenRepository),
      new CookieClearingLogoutHandler("cookiesToClear", "XSRF-TOKEN")
    )
    logoutFilter.setFilterProcessesUrl("/xldeploy/logout")
    logoutFilter
  }

}
