package com.xebialabs.deployit.plugin.remoting.inspection;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.ITask;
import com.xebialabs.deployit.plugin.api.inspection.InspectionContext;
import com.xebialabs.deployit.plugin.api.reflect.*;
import com.xebialabs.deployit.plugin.api.services.Repository;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.UnsupportedEncodingException;
import java.util.Arrays;
import java.util.Collection;
import java.util.Map;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.CI;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.LIST_OF_CI;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.SET_OF_CI;
import static java.lang.String.format;
import static java.net.URLDecoder.decode;

public class InspectionProtocolContext implements ExecutionContext {

    private final ExecutionContext context;
    private static final String defaultEncoding = "UTF-8";

    public static final String DISCOVERED_ITEM_PRELUDE = "DISCOVERED-ITEM:";
    public static final String INSPECTED_PROPERTY_PRELUDE = "INSPECTED-PROPERTY:";
    public static final String INSPECTED_ITEM_PRELUDE = "INSPECTED-ITEM";

    protected ConfigurationItem currentInspectedItem;

    private volatile boolean inspectionFailed;


    public InspectionProtocolContext(ConfigurationItem inspectedItem,ExecutionContext context) {
        this.context = context;
        this.currentInspectedItem = inspectedItem;
    }

    public boolean isInspectionFailed() {
        return inspectionFailed;
    }

    @Override
    public void logOutput(String output) {
        try {
            logger.trace("Handling inspection protocol line {}", output);
            if (handleOutputLine(output, context)) {
                logger.trace("Handled inspection protocol line {}", output);
            } else {
                context.logOutput(output);
            }
        } catch(RuntimeException exc) {
            context.logError("Inspection failed", exc);
            inspectionFailed = true;
        }
    }

    protected boolean handleOutputLine(String line, ExecutionContext ctx) {
        String[] lineSegments = line.split(":");

        try {
            if (line.startsWith(DISCOVERED_ITEM_PRELUDE)) {
                checkArgument(lineSegments.length == 3, "%s is incorrectly encoded. Expected format '%sid:type' but was %s",DISCOVERED_ITEM_PRELUDE,DISCOVERED_ITEM_PRELUDE,line);
                String id = decode(lineSegments[1], defaultEncoding);
                String typeName = lineSegments[2];
                return handleDiscoveredItem(id, typeName, ctx);
            }

            if (line.startsWith(INSPECTED_PROPERTY_PRELUDE)) {
                checkArgument(lineSegments.length > 2, "%s is incorrectly encoded. Expected format '%sid:name:value1:value2:..' but was %s",INSPECTED_PROPERTY_PRELUDE,INSPECTED_PROPERTY_PRELUDE,line);
                String id = decode(lineSegments[1], defaultEncoding);
                String name = decode(lineSegments[2], defaultEncoding);
                String[] values = new String[lineSegments.length - 3];
                System.arraycopy(lineSegments, 3, values, 0, values.length);
                for (int i = 0; i < values.length; i++) {
                    values[i] = decode(values[i], defaultEncoding);
                }
                return handleInspectedProperty(id, name, values, ctx);
            }

            if (line.startsWith(INSPECTED_ITEM_PRELUDE)) {
                checkArgument(lineSegments.length == 2, "%s is incorrectly encoded. Expected format '%sid' but was %s",INSPECTED_ITEM_PRELUDE,line);
                String id = decode(lineSegments[1], defaultEncoding);
                return handleInspectedItem(id, ctx);
            }

        } catch (UnsupportedEncodingException encodingException) {
            logger.debug("Unsupported encoding exception was encountered while parsing inspected and / or discovered items");
            return false;
        }
        return false;
    }

    private boolean handleDiscoveredItem(String id, String typeName, ExecutionContext ctx) throws UnsupportedEncodingException {
        Type type = Type.valueOf(decode(typeName, defaultEncoding));
        if (!DescriptorRegistry.exists(type)) {
            ctx.logError("Discovered item " + id + " of unknown type " + type);
            return false;
        }
        Descriptor d = DescriptorRegistry.getDescriptor(type);
        ConfigurationItem discoveredItem = d.newInstance(id);

        setReferenceFromChildToParent(discoveredItem, d, ctx);

        ctx.getInspectionContext().discovered(discoveredItem);
        logger.debug("Discovered item {} with type {}", id, type);
        return true;
    }

    @SuppressWarnings("unchecked")
    protected void setReferenceFromChildToParent(ConfigurationItem discoveredItem, Descriptor d, ExecutionContext ctx) {
        for (PropertyDescriptor pd : d.getPropertyDescriptors()) {
            boolean isReferenceToParent = pd.getKind() == CI && pd.isAsContainment();
            if (isReferenceToParent) {
                ConfigurationItem parent = getParent(discoveredItem.getId(), pd, ctx);
                pd.set(discoveredItem, parent);

                for(PropertyDescriptor ppd : parent.getType().getDescriptor().getPropertyDescriptors()) {
                    boolean isReferenceToChild = (ppd.getKind() == SET_OF_CI || ppd.getKind() == LIST_OF_CI) && ppd.isAsContainment() && d.isAssignableTo(ppd.getReferencedType());
                    if(isReferenceToChild) {
                        Collection<ConfigurationItem> refs = (Collection<ConfigurationItem>) ppd.get(parent);
                        refs.add(discoveredItem);
                        ppd.set(parent, refs);
                    }
                }
            }
        }
    }

    private ConfigurationItem getParent(String id, PropertyDescriptor pd, ExecutionContext ctx) {
        String parentId = getParentId(id);
        ConfigurationItem parent;
        if(parentId.equals(currentInspectedItem.getId())) {
            parent = currentInspectedItem;
        } else {
            parent = ctx.getInspectionContext().getInspected().get(parentId);
            if(parent == null) {
                parent = ctx.getInspectionContext().getDiscovered().get(parentId);
                checkArgument(parent != null, "Cannot resolve parent reference from item [%s] for property [%s]", id, pd);
            }
        }
        return parent;
    }

    private static String getParentId(String id) {
        int indexOfLastSlash = id.lastIndexOf('/');
        checkArgument(indexOfLastSlash != -1, "[%s] has no parent", id);
        return id.substring(0, indexOfLastSlash);
    }

    @SuppressWarnings("unchecked")
    private boolean handleInspectedProperty(String id, String name, String[] values, ExecutionContext ctx) {
        logger.trace("Handling inspection for property {} on ci {} with values {}", new Object[] {name, id, Arrays.toString(values)});

        if (values.length == 0) {
            logger.debug("No inspected value specified for property {} on ci {}. Ignoring.", name, id);
            return false;
        }

        ConfigurationItem inspectedItem;
        if(id.equals("this")) {
            inspectedItem = currentInspectedItem;
        } else {
            inspectedItem = ctx.getInspectionContext().getDiscovered().get(id);
            if(inspectedItem == null) {
                ctx.logError(format("Cannot inspect property %s on unknown item %s", name, id));
                return false;
            }
        }

        Descriptor d = inspectedItem.getType().getDescriptor();
        PropertyDescriptor pd = d.getPropertyDescriptor(name);
        if (pd == null) {
            ctx.logOutput("Inspected unknown property " + name + ". Ignoring...");
            return false;
        }

        final PropertyKind kind = pd.getKind();

        if (pd.isHidden()) {
            ctx.logOutput("Inspected property " + name + " is hidden. Ignoring...");
            return false;
        }

        switch (kind) {
            case SET_OF_CI:
            case LIST_OF_CI:
                Collection<ConfigurationItem> refs = checkNotNull((Collection<ConfigurationItem>) pd.get(inspectedItem));
                refs.add(resolveCiReference(values[0], pd, ctx));
                break;
            case SET_OF_STRING:
            case LIST_OF_STRING:
                Collection<String> stringProperties = checkNotNull((Collection < String >) pd.get(inspectedItem));
                stringProperties.add(values[0]);
                break;
            case MAP_STRING_STRING:
                Map<String, String> mapProperties = checkNotNull(((Map<String, String>) pd.get(inspectedItem)));
                if(values.length == 1) {
                    mapProperties.put(values[0], "");
                } else {
                    mapProperties.put(values[0],values[1]);
                }
                break;
            case CI:
                pd.set(inspectedItem, resolveCiReference(values[0], pd, ctx));
                break;
            default:
                pd.set(inspectedItem, values[0]);
        }

        logger.debug("Inspected property {} on item {}", name, id);
        return true;
    }

    private static ConfigurationItem resolveCiReference(String id, PropertyDescriptor pd, ExecutionContext ctx) {
        ConfigurationItem referencedCi = ctx.getInspectionContext().getInspected().get(id);
        checkArgument(referencedCi != null, "Cannot resolve CI reference to [%s] from [%s]", id, pd);
        return referencedCi;
    }

    private static boolean handleInspectedItem(String id, ExecutionContext ctx) {
        ConfigurationItem inspectedItem = ctx.getInspectionContext().getDiscovered().get(id);
        if(inspectedItem == null) {
            ctx.logError(format("Cannot mark %s as inspected because it has not been discovered yet", id));
            return false;
        }

        logger.debug("Inspected item {}", inspectedItem);
        ctx.getInspectionContext().inspected(inspectedItem);

        return true;
    }


    @Override
    public void logError(String error) {
        context.logError(error);
    }

    @Override
    public void logError(String error, Throwable t) {
        context.logError(error, t);
    }

    @Override
    public Object getAttribute(String name) {
        return context.getAttribute(name);
    }

    @Override
    public void setAttribute(String name, Object value) {
        context.setAttribute(name, value);
    }

    @Override
    public Repository getRepository() {
        return context.getRepository();
    }

    @Override
    public InspectionContext getInspectionContext() {
        return context.getInspectionContext();
    }

    @Override
    public ITask getTask() {
        return context.getTask();
    }

    private static Logger logger = LoggerFactory.getLogger(InspectionProtocolContext.class);
}
