package com.xebialabs.deployit.plugin.overthere.upgrade;

import java.util.List;

import javax.jcr.Node;
import javax.jcr.RepositoryException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.server.api.repository.RawRepository;
import com.xebialabs.deployit.server.api.upgrade.Upgrade;
import com.xebialabs.deployit.server.api.upgrade.UpgradeException;
import com.xebialabs.deployit.server.api.upgrade.Version;
import com.xebialabs.overthere.cifs.CifsConnectionBuilder;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.overthere.cifs.CifsConnectionBuilder.CONNECTION_TYPE;
import static com.xebialabs.overthere.cifs.CifsConnectionBuilder.WINRM_ENABLE_HTTPS;

public class UpgradeToRemotingPlugin394 extends Upgrade {

    @Override
    public Version upgradeVersion() {
        return Version.valueOf("remoting-plugin", "3.9.4");
    }

    @Override
    public boolean doUpgrade(RawRepository repository) throws UpgradeException {
        try {
            convertCifsConnectionTypes(repository);
            return true;
        } catch (RepositoryException exc) {
            throw new UpgradeException("Cannot upgrade repository to format for " + upgradeVersion(), exc);
        }
    }

    private void convertCifsConnectionTypes(RawRepository repository) throws RepositoryException {
        for(Type t: getAssignmentCompatibleTypes("overthere.CifsHost")) {
            List<Node> websiteSpecs = repository.findNodesByType(t);
            for (Node websiteSpec : websiteSpecs) {
                convertCifsConnectionType(repository, websiteSpec);
            }
         }
    }

    private List<Type> getAssignmentCompatibleTypes(String typeName) {
        Type t = Type.valueOf(typeName);
        List<Type> types = newArrayList(DescriptorRegistry.getSubtypes(t));
        types.add(t);
        return types;
    }

    private void convertCifsConnectionType(RawRepository repository, Node overthereCifsHost) throws RepositoryException {
        if (overthereCifsHost.hasProperty(CifsConnectionBuilder.CONNECTION_TYPE)) {
            String updatedConnectionType = null;
            Boolean updatedWinrmEnableHttps = null;

            String ct = overthereCifsHost.getProperty(CifsConnectionBuilder.CONNECTION_TYPE).getString();
            if(ct.equals("WINRM")) {
                updatedConnectionType = "WINRM_INTERNAL";
            } else if(ct.equals("WINRM_HTTP")) {
                updatedConnectionType = "WINRM_INTERNAL";
                updatedWinrmEnableHttps = false;
            } else if(ct.equals("WINRM_HTTPS")) {
                updatedConnectionType = "WINRM_INTERNAL";
                updatedWinrmEnableHttps = true;
            } else if(ct.equals("WINRS")) {
                updatedConnectionType = "WINRM_NATIVE";
            }
            
            if(updatedConnectionType != null) {
                logger.info("Converting property " + CONNECTION_TYPE + " on CI [{}] from {} to {}", overthereCifsHost.getPath().substring(1), ct, updatedConnectionType);
                overthereCifsHost.setProperty(CONNECTION_TYPE, updatedConnectionType);
            }

            if(updatedWinrmEnableHttps != null) {
                logger.info("Setting property " + WINRM_ENABLE_HTTPS + " on CI [{}] to {}", overthereCifsHost.getPath().substring(1), updatedWinrmEnableHttps);
                overthereCifsHost.setProperty(WINRM_ENABLE_HTTPS, updatedWinrmEnableHttps);
            }
        }
    }

    private static Logger logger = LoggerFactory.getLogger(UpgradeToRemotingPlugin394.class);
}
