package com.xebialabs.deployit.plugin.remoting.inspection;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.udm.Container;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Supplier;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;

import com.xebialabs.deployit.plugin.api.inspection.InspectionContext;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;


import static com.google.common.collect.Maps.newHashMap;

@SuppressWarnings("rawtypes")
public abstract class DeployedInspectionHelper<C extends Container, D extends Deployed> {

    public static final String DISCOVER_SCRIPT_PROPERTY_NAME = "discoverScript";
    private Class<C> containerSuperClass;
    private Class<D> deployedSuperClass;

    public DeployedInspectionHelper(Class<C> containerSuperClass, Class<D> deployedSuperClass) {
        this.containerSuperClass = containerSuperClass;
        this.deployedSuperClass = deployedSuperClass;
    }

    public void discoverDeployeds(C container, InspectionContext ctx) {
        discoverDeployeds(container, ctx, DescriptorRegistry.getSubtypes(Type.valueOf(deployedSuperClass)));
    }

    public void discoverDeployeds(C container, InspectionContext ctx, Iterable<Type> types) {
        Multimap<Integer, Step> steps = newSortedKeyMultimapForSteps();
        final Type containerT = container.getType();
        for (Type deployedT : types) {
            final Descriptor deployedD = deployedT.getDescriptor();
            logger.debug("Checking whether type [{}] is not virtual", deployedT);
            if (deployedD.isVirtual()) {
                continue;
            }

            final Type deployedContainerT = deployedD.getContainerType();
            final Type targetContainerT = Type.valueOf(containerSuperClass);
            logger.debug("Checking whether type [{}] is instance of type [{}]", deployedContainerT, targetContainerT);
            if (!containerT.instanceOf(targetContainerT)) {
                continue;
            }

            logger.debug("Checking whether type [{}] has a discoveryScript property", deployedT);
            final PropertyDescriptor discoveryScriptProperty = deployedD.getPropertyDescriptor(DISCOVER_SCRIPT_PROPERTY_NAME);
            if (discoveryScriptProperty == null) {
                continue;
            }

            logger.debug("Checking whether the " + DISCOVER_SCRIPT_PROPERTY_NAME + " property of type [{}] has a default value", deployedT);
            final Object defaultValue = discoveryScriptProperty.getDefaultValue();
            if (defaultValue == null) {
                continue;
            }

            logger.debug("Checking whether the " + DISCOVER_SCRIPT_PROPERTY_NAME + " property of type [{}] has a string value", deployedD.getType());
            if (!(defaultValue instanceof String)) {
                continue;
            }

            final String scriptName = (String) defaultValue;
            logger.debug("Adding inspection step using script {} for container {}", scriptName, container);
            final Map<String, Object> varsCtx = newHashMap();
            varsCtx.put("container", container);
            D prototype = deployedD.newInstance();
            varsCtx.put("prototype", prototype);
            Step step = createInspectionStep(container, prototype, scriptName, varsCtx, "Discover objects of type " + deployedT + " on " + container);

            int discoverOrder = 0;
            if (prototype.hasProperty("discoverOrder")) {
                discoverOrder = prototype.<Integer>getProperty("discoverOrder");
            }

            steps.put(discoverOrder, step);
        }

        for (Step step : steps.values()) {
            ctx.addStep(step);
        }
    }

    protected abstract Step createInspectionStep(C container, D prototype, String scriptName, Map<String,Object> varsCtx, String description);

    private Multimap<Integer, Step> newSortedKeyMultimapForSteps() {
        return Multimaps.newListMultimap(new TreeMap<Integer, Collection<Step>>(), new Supplier<List<Step>>() {
            public List<Step> get() {
                return Lists.newArrayList();
            }
        });
    }

    private static Logger logger = LoggerFactory.getLogger(DeployedInspectionHelper.class);

}
