package com.xebialabs.deployit.plugin.remoting.scripts;

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.charset.Charset;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import org.apache.commons.io.ByteOrderMark;
import org.apache.commons.io.input.BOMInputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.io.CharStreams;

import com.xebialabs.overthere.OverthereConnection;
import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.overthere.RuntimeIOException;
import com.xebialabs.overthere.util.OverthereUtils;

import de.schlichtherle.truezip.file.TFile;
import de.schlichtherle.truezip.file.TFileInputStream;

import static com.xebialabs.overthere.util.OverthereUtils.getName;
import static org.apache.commons.io.IOUtils.closeQuietly;

public class ScriptUtils {

    public static final String MDC_KEY_SCRIPT_PATH = "scriptPath";
    
    public static final ByteOrderMark[] detectedBOMs = new ByteOrderMark[] { ByteOrderMark.UTF_8, ByteOrderMark.UTF_16BE, ByteOrderMark.UTF_16LE };

    public static void appendScripts(List<String> scriptPaths, StringBuilder b) {
        for(String s: scriptPaths) {
            appendScript(s, b);
        }
    }

    public static void appendScript(String scriptPath, StringBuilder b) {
        b.append(loadScript(scriptPath));
    }
    
    public static void appendScriptDir(String scriptDirPath, StringBuilder b) {
        b.append(loadScriptDir(scriptDirPath));
    }

    public static String loadScript(String scriptPath) {
        logger.debug("Loading script from [{}]", scriptPath);

        URL scriptUrl = Thread.currentThread().getContextClassLoader().getResource(scriptPath);
        if (scriptUrl == null) {
            throw new IllegalArgumentException("Cannot load script from " + scriptPath + ": script does not exist on classpath");
        }

        TFile scriptFile = null;
        try {
            scriptFile = new TFile(new URI(scriptUrl.toString()));
            return loadScriptFromFile(scriptFile);
        } catch (URISyntaxException exc) {
            throw new RuntimeIOException("Cannot load script from " + scriptPath, exc);
        } catch (IOException exc) {
            throw new RuntimeIOException("Cannot load script from " + scriptPath, exc);
        } finally {
            umountQuietly(scriptFile);
        }
    }

    public static String loadScriptDir(String scriptDirPath) {
        if(scriptDirPath == null || scriptDirPath.length() == 0) {
            return "";
        }

        logger.debug("Loading scripts from [{}]", scriptDirPath);

        URL scriptDirUrl = Thread.currentThread().getContextClassLoader().getResource(scriptDirPath);
        if (scriptDirUrl == null) {
            throw new IllegalArgumentException("Cannot load scripts from " + scriptDirPath + ": script directory does not exist on classpath");
        }

        TFile scriptDir = null;
        try {
            scriptDir = new TFile(new URI(scriptDirUrl.toString()));
            TFile[] scriptDirFiles = scriptDir.listFiles();
            if (scriptDirFiles == null)
                throw new IllegalArgumentException("Cannot load scripts from " + scriptDir + ": not a directory");

            StringBuilder b = new StringBuilder();
            for (TFile scriptFile : scriptDirFiles) {
                b.append(loadScriptFromFile(scriptFile));
            }
            return b.toString();
        } catch (URISyntaxException exc) {
            throw new RuntimeIOException("Cannot read scripts from " + scriptDirPath, exc);
        } catch (IOException exc) {
            throw new RuntimeIOException("Cannot read scripts from " + scriptDirPath, exc);
        } finally {
            umountQuietly(scriptDir);
        }
    }

    public static String loadScriptFromFile(TFile scriptFile) throws IOException {
        BOMInputStream scriptInputStream = new BOMInputStream(new TFileInputStream(scriptFile), detectedBOMs);
        Reader scriptReader;
        if(scriptInputStream.hasBOM()) {
            String charsetName = scriptInputStream.getBOMCharsetName();
            logger.debug("Reading script from [{}] with encoding [{}]", scriptFile, charsetName);
            Charset charset = Charset.forName(charsetName);
            scriptReader = new InputStreamReader(scriptInputStream, charset);
        } else {
            logger.debug("Reading script from [{}] with default platform encoding", scriptFile);
            scriptReader = new InputStreamReader(scriptInputStream);
        }

        try {  
            try {
                return "# " + scriptFile + "\n" + CharStreams.toString(scriptReader);
            } catch (IOException exc) {
                throw new IllegalArgumentException("Cannot load script from " + scriptFile, exc);
            }
        } finally {
            closeQuietly(scriptReader);
        }
    }

    public static void dumpScript(String scriptTemplateName, String evaluatedScriptContents, Logger logger) {
        if (logger.isTraceEnabled()) {
            DateFormat df = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS");
            logger.trace(":::: "  + df.format(new Date()) + " :::: " + scriptTemplateName + " ::::");
            String[] lines = evaluatedScriptContents.split("\r?\n");
            for (int i = 0; i < lines.length; i++) {
                logger.trace((i + 1) + " : " + lines[i]);
            }
        }
    }

    public static OverthereFile uploadScript(OverthereConnection connection, String scriptFileNameTemplate, String scriptContents) {
        OverthereFile uploadedFile = connection.getTempFile(getName(scriptFileNameTemplate));
        OverthereUtils.write(scriptContents, "UTF-8", uploadedFile);
        return uploadedFile;
    }

    static void umountQuietly(TFile file) {
        if (file != null && file.isArchive() && file.getEnclArchive() == null) {
            try {
                TFile.umount(file);
            } catch (Exception e) {
                logger.error("Couldn't umount [{}], ignoring exception.", file);
                logger.debug("Exception while umounting was: ", e);
            }
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(ScriptUtils.class);
}
