package com.xebialabs.deployit.plugin.overthere.step;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.overthere.DefaultExecutionOutputHandler;
import com.xebialabs.deployit.plugin.overthere.Host;
import com.xebialabs.overthere.OverthereConnection;
import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.overthere.util.OverthereUtils;

import static com.xebialabs.deployit.plugin.overthere.DefaultExecutionOutputHandler.handleStderr;
import static com.xebialabs.deployit.plugin.overthere.DefaultExecutionOutputHandler.handleStdout;
import static com.xebialabs.overthere.ConnectionOptions.ADDRESS;
import static org.apache.commons.io.IOUtils.closeQuietly;

@SuppressWarnings("serial")
public class CheckFileTransferStep implements Step {

    private final Host host;

    private final String hostname;

    public CheckFileTransferStep(Host host) {
        this.host = host;
        if (host.hasProperty(ADDRESS)) {
            this.hostname = host.getProperty(ADDRESS);
        } else {
            this.hostname = host.getName();
        }
    }

    @Override
    public String getDescription() {
        return "Check whether XL Deploy can transfer files to " + hostname;
    }

    @Override
    public int getOrder() {
        return 50;
    }

    @Override
    public StepExitCode execute(ExecutionContext ctx) throws Exception {
        DefaultExecutionOutputHandler stdoutHandler = handleStdout(ctx);
        DefaultExecutionOutputHandler stderrHandler = handleStderr(ctx);
        try {
            try {
                OverthereConnection connection = host.getConnection();
                try {
                    ctx.logOutput("Uploading file to the temporary directory...");
                    OverthereFile uploadedFile = uploadFile(connection);
                    ctx.logOutput("Checking read permission of uploaded file...");
                    tryCanGetFilePermissions(uploadedFile);
                } finally {
                    closeQuietly(connection);
                }
            } catch (Exception exc) {
                ctx.logError("Cannot transfer files to " + hostname + ". Please check the connection settings.", exc);
                return StepExitCode.FAIL;
            }
            ctx.logOutput("Successfully transferred files to " + hostname + ".");
            return StepExitCode.SUCCESS;
        } finally {
            closeQuietly(stdoutHandler);
            closeQuietly(stderrHandler);
        }
    }

    private static OverthereFile uploadFile(OverthereConnection connection) {
        OverthereFile fileToUpload = connection.getTempFile("hostconnection-remote", ".txt");
        OverthereUtils.write("Contents of host connection test file", "UTF-8", fileToUpload);
        return fileToUpload;
    }

    private static void tryCanGetFilePermissions(OverthereFile uploadedFile) {
        if (!uploadedFile.canRead()) {
            throw new RuntimeException("Cannot check read permission for file just uploaded");
        }
    }
}