/*
 * Copyright (c) 2008-2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin.overthere;

import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Lists;

import com.xebialabs.deployit.plugin.api.execution.Step;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.ControlTask;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.base.BaseContainer;
import com.xebialabs.deployit.plugin.overthere.step.CheckConnectionStep;
import com.xebialabs.overthere.ConnectionOptions;
import com.xebialabs.overthere.OperatingSystemFamily;
import com.xebialabs.overthere.Overthere;
import com.xebialabs.overthere.OverthereConnection;

import static com.google.common.base.Strings.isNullOrEmpty;
import static com.google.common.base.Strings.nullToEmpty;
import static com.xebialabs.deployit.plugin.overthere.LegacyCifsConnectionType.WINRM_HTTP;
import static com.xebialabs.deployit.plugin.overthere.LegacyCifsConnectionType.WINRM_HTTPS;
import static com.xebialabs.overthere.ConnectionOptions.TEMPORARY_DIRECTORY_PATH;
import static com.xebialabs.overthere.cifs.CifsConnectionBuilder.CONNECTION_TYPE;
import static com.xebialabs.overthere.cifs.CifsConnectionBuilder.WINRM_ENABLE_HTTPS;
import static com.xebialabs.overthere.cifs.CifsConnectionType.TELNET;
import static com.xebialabs.overthere.cifs.CifsConnectionType.WINRM;
import static java.lang.String.format;

/**
 * Base class for machines that runs middleware.
 */
@SuppressWarnings("serial")
@Metadata(root = Metadata.ConfigurationItemRoot.INFRASTRUCTURE, virtual = true, description = "Machine that runs middleware, on which scripts can be executed, etc.")
public class Host extends BaseContainer implements HostContainer {

    @Property(description = "Protocol to use when connecting to this host", hidden = true)
    private String protocol;

    @Property(description = "Operating system")
    private OperatingSystemFamily os;

    @Property(category = "Advanced", description = "Directory into which temporary files are stored. Will be cleaned up when the connection is closed.", required = false)
    private String temporaryDirectoryPath;

    public OverthereConnection getConnection() {
        logger.debug("Using connection protocol {}", protocol);
        return Overthere.getConnection(protocol, getConnectionOptions());
    }

    private ConnectionOptions getConnectionOptions() {
        ConnectionOptions options = new ConnectionOptions();

        copyPropertiesToConnectionOptions(options, this);

        // set the temporary directory path if it has been set
        if (!nullToEmpty(getTemporaryDirectoryPath()).trim().isEmpty()) {
            setConnectionOption(options, TEMPORARY_DIRECTORY_PATH, getTemporaryDirectoryPath());
        }

        logger.debug("Created connection options: {}", options);

        return options;
    }

    protected void copyPropertiesToConnectionOptions(ConnectionOptions options,
        ConfigurationItem ci) {
        // copy all CI properties to connection properties
        for (PropertyDescriptor pd : ci.getType().getDescriptor()
            .getPropertyDescriptors()) {
            Object value = pd.get(ci);
            setConnectionOption(options, pd.getName(), value);
        }

        fixLegacyCifsConnectionType(options);
    }

    private void setConnectionOption(ConnectionOptions options, String key,
        Object value) {
        if (key.equals("protocol") || key.equals("temporaryDirectoryPath")) {
            return;
        }
        if (value == null || isNullOrEmpty(value.toString())) {
            return;
        }

        if (key.equals("jumpstation")) {
            ConnectionOptions jumpstationOptions = new ConnectionOptions();
            copyPropertiesToConnectionOptions(jumpstationOptions,
                (ConfigurationItem) value);
            options.set(key, jumpstationOptions);
        } else {
            options.set(key, value);
        }
    }

    private void fixLegacyCifsConnectionType(ConnectionOptions options) {
        Object connectionType = options.getOptional(CONNECTION_TYPE);
        if (connectionType instanceof LegacyCifsConnectionType) {
            switch ((LegacyCifsConnectionType) connectionType) {
            case TELNET:
                connectionType = TELNET;
                break;
            case WINRM:
                connectionType = WINRM;
                break;
            case WINRM_HTTP:
                logger.warn("Translating legacy CIFS connection type " + WINRM_HTTP + " to new type " + WINRM + " and setting " + WINRM_ENABLE_HTTPS + " option to false");
                connectionType = WINRM;
                options.set(WINRM_ENABLE_HTTPS, false);
                break;
            case WINRM_HTTPS:
                logger.warn("Translating legacy CIFS connection type " + WINRM_HTTPS + " to new type " + WINRM + " and setting " + WINRM_ENABLE_HTTPS + " option to true");
                connectionType = WINRM;
                options.set(WINRM_ENABLE_HTTPS, true);
                break;
            default:
                throw new IllegalArgumentException(format(
                    "Unknown value [%s] for property [%s] of type "
                        + LegacyCifsConnectionType.class.getName(),
                    connectionType, connectionType));
            }
            options.set(CONNECTION_TYPE, connectionType);
        }
    }

    @ControlTask(description = "Check connection")
    public List<Step<?>> checkConnection() {
        return Lists.<Step<?>> newArrayList(new CheckConnectionStep(this));
    }

    @Override
    public Host getHost() {
        return this;
    }

    public OperatingSystemFamily getOs() {
        return os;
    }

    public void setOs(OperatingSystemFamily os) {
        this.os = os;
    }

    public String getTemporaryDirectoryPath() {
        return temporaryDirectoryPath;
    }

    public void setTemporaryDirectoryPath(String temporaryDirectoryPath) {
        this.temporaryDirectoryPath = temporaryDirectoryPath;
    }

    private static Logger logger = LoggerFactory.getLogger(Host.class);

}
