package com.xebialabs.deployit.plugin.remoting.scripts;

import static com.xebialabs.overthere.util.OverthereUtils.getName;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.Reader;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;

import org.slf4j.Logger;

import com.google.common.io.CharStreams;
import com.google.common.io.Closeables;
import com.xebialabs.deployit.plugin.api.utils.TFileUtils;
import com.xebialabs.overthere.OverthereConnection;
import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.overthere.RuntimeIOException;
import com.xebialabs.overthere.util.OverthereUtils;

import de.schlichtherle.truezip.file.TFile;
import de.schlichtherle.truezip.file.TFileReader;

public class ScriptUtils {

	public static final String MDC_KEY_SCRIPT_PATH = "scriptPath";

	public static String loadScript(String scriptPath) {
		URL scriptUrl = Thread.currentThread().getContextClassLoader().getResource(scriptPath);
		if (scriptUrl == null) {
			throw new IllegalArgumentException("Cannot load script from " + scriptPath + ": script does not exist on classpath");
		}

		TFile scriptFile = null;
		try {
			scriptFile = new TFile(new URI(scriptUrl.toString()));
			return loadScriptFromFile(scriptFile);
		} catch (URISyntaxException exc) {
			throw new RuntimeIOException("Cannot load script from " + scriptPath, exc);
		} catch (IOException exc) {
			throw new RuntimeIOException("Cannot load script from " + scriptPath, exc);
		} finally {
			TFileUtils.umountQuietly(scriptFile);
		}
	}

	public static String loadScriptDir(String scriptDirPath) {
		if(scriptDirPath == null) {
			return "";
		}

		URL scriptDirUrl = Thread.currentThread().getContextClassLoader().getResource(scriptDirPath);
		if (scriptDirUrl == null) {
			throw new IllegalArgumentException("Cannot load scripts from " + scriptDirPath + ": script directory does not exist on classpath");
		}

		TFile scriptDir = null;
		try {
			scriptDir = new TFile(new URI(scriptDirUrl.toString()));
			TFile[] scriptDirFiles = scriptDir.listFiles();
			if (scriptDirFiles == null)
				throw new IllegalArgumentException("Cannot load scripts from " + scriptDir + ": not a directory");

			StringBuilder b = new StringBuilder();
			for (TFile scriptFile : scriptDirFiles) {
				b.append(loadScriptFromFile(scriptFile));
			}
			return b.toString();
		} catch (URISyntaxException exc) {
			throw new RuntimeIOException("Cannot read scripts from " + scriptDirPath, exc);
		} catch (IOException exc) {
			throw new RuntimeIOException("Cannot read scripts from " + scriptDirPath, exc);
		} finally {
			TFileUtils.umountQuietly(scriptDir);
		}
	}

	public static String loadScriptFromFile(TFile scriptFile) throws FileNotFoundException {
		Reader scriptReader = new TFileReader(scriptFile);
		try {
			try {
				return "# " + scriptFile + "\n" + CharStreams.toString(scriptReader);
			} catch (IOException exc) {
				throw new IllegalArgumentException("Cannot load script from " + scriptFile, exc);
			}
		} finally {
			Closeables.closeQuietly(scriptReader);
		}
	}

	public static void dumpScript(String scriptTemplateName, String evaluatedScriptContents, Logger logger) {
		if (logger.isTraceEnabled()) {
			DateFormat df = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS");
			logger.trace(":::: "  + df.format(new Date()) + " :::: " + scriptTemplateName + " ::::");
			String[] lines = evaluatedScriptContents.split("\r?\n");
			for (int i = 0; i < lines.length; i++) {
				logger.trace((i + 1) + " : " + lines[i]);
			}
		}
	}

	public static OverthereFile uploadScript(OverthereConnection connection, String scriptFileNameTemplate, String scriptContents) {
		OverthereFile uploadedFile = connection.getTempFile(getName(scriptFileNameTemplate));
		OverthereUtils.write(scriptContents, "UTF-8", uploadedFile);
		return uploadedFile;
	}

}
